<?php

use Livewire\Component;
use Illuminate\Validation\Rule;
use App\Enums\LoanOptions;
use App\Support\Currency;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Transaction;
use App\Support\LoanCalculator;
use App\Notifications\TransactionNotification;
use App\Services\AdminNotifier;
new class extends Component {
    public $accounts = [];
    public $selectedAccount = null;
    public $currencySymbol;
    public $amount = '';
    public $duration = '';
    public $credit_facility = '';
    public $employment_status = '';
    public $net_income = '';
    public $purpose = '';
    public $minAmount = 300;
    public $agreement_terms = false;
    public $agreement_privacy = false;
    public $agreement_credit = false;
    public string $idempotencyKey = '';
    public string $confirmedAmount = '0';
    public string $confirmedDuration = '0';
    public string $confirmedRate = '0';
    public string $confirmedTotalInterest = '0';
    public string $confirmedMonthlyRepayment = '0';
    public ?int $confirmedAccountId = null;

    #[Computed]
    public function durations()
    {
        return LoanOptions::durations();
    }

    #[Computed]
    public function creditFacilities()
    {
        return LoanOptions::creditFacilities();
    }

    #[Computed]
    public function employmentStatuses()
    {
        return LoanOptions::employmentStatuses();
    }

    #[Computed]
    public function netIncomeRanges()
    {
        return LoanOptions::netIncomeRanges();
    }

    #[Computed]
    public function monthlyRepayment()
    {
        $amount = preg_replace('/[^\d.]/', '', $this->amount);
        $duration = preg_replace('/[^\d]/', '', $this->duration);

        if (bccomp($amount, '0', 2) <= 0 || bccomp($duration, '0', 0) <= 0 || !$this->credit_facility) {
            return '0.00';
        }

        $rate = (string) config("loan.interest_rates.{$this->credit_facility}", '0.04');

        $interestPerMonth = bcmul($amount, $rate, 4);
        $totalInterest = bcmul($interestPerMonth, $duration, 2);
        $totalRepayment = bcadd($amount, $totalInterest, 2);

        return bcdiv($totalRepayment, $duration, 2);
    }

    #[Computed]
    public function repaymentSchedule(): array
    {
        $amount = preg_replace('/[^\d.]/', '', $this->amount);
        $duration = preg_replace('/[^\d]/', '', $this->duration);

        if (bccomp($amount, '0', 2) <= 0 || bccomp($duration, '0', 0) <= 0 || !$this->credit_facility) {
            return [];
        }

        $rate = (string) config("loan.interest_rates.{$this->credit_facility}", '0.04');

        $schedule = [];

        // Calculate monthly amortized payment: M = P * [ r*(1+r)^n ] / [ (1+r)^n - 1 ]
        $r = $rate; // monthly interest rate
        $n = $duration; // number of months

        // (1+r)^n
        $onePlusR = bcadd('1', $r, 8);
        $onePlusRToN = bcpow($onePlusR, $n, 8);

        // numerator: P * r * (1+r)^n
        $numerator = bcmul(bcmul($amount, $r, 8), $onePlusRToN, 8);

        // denominator: (1+r)^n - 1
        $denominator = bcsub($onePlusRToN, '1', 8);

        // Monthly payment
        $monthlyPayment = bcdiv($numerator, $denominator, 2);

        $remainingPrincipal = $amount;

        for ($i = 1; $i <= (int) $duration; $i++) {
            // Interest for this month = remaining principal * rate
            $interest = bcmul($remainingPrincipal, $r, 2);

            // Principal = monthly payment – interest
            $principal = bcsub($monthlyPayment, $interest, 2);

            // Update remaining principal
            $remainingPrincipal = bcsub($remainingPrincipal, $principal, 2);

            // Correct last month rounding difference
            if ($i === (int) $duration && bccomp($remainingPrincipal, '0', 2) !== 0) {
                $principal = bcadd($principal, $remainingPrincipal, 2);
                $monthlyPayment = bcadd($principal, $interest, 2);
                $remainingPrincipal = '0.00';
            }

            $schedule[] = [
                'month' => $i,
                'principal' => $principal,
                'interest' => $interest,
                'total' => $monthlyPayment,
            ];
        }

        return $schedule;
    }

    /* =========================
    |  Lifecycle
    ========================= */
    public function mount()
    {
        $user = auth()->user();

        $this->accounts = $user->profile ? $user->profile->accounts()->where('is_active', true)->get() : collect();

        $this->selectedAccount = $this->accounts->first()?->id;

        $this->currencySymbol = $user->profile ? Currency::symbol($user->profile->currency) : '$';

        $this->idempotencyKey = (string) Str::uuid();
    }

    protected function rules(): array
    {
        return [
            'amount' => 'required|numeric',
            'duration' => ['required', Rule::in(LoanOptions::durations())],
            'credit_facility' => ['required', Rule::in(LoanOptions::creditFacilities())],
            'employment_status' => ['required', Rule::in(LoanOptions::employmentStatuses())],
            'net_income' => ['required', Rule::in(LoanOptions::netIncomeRanges())],
            'purpose' => 'required|string',
            'agreement_terms' => 'accepted',
            'agreement_privacy' => 'accepted',
            'agreement_credit' => 'accepted',
        ];
    }

    public function confirmLoan()
    {
        $this->validate();

        $user = auth()->user();
        $account = $this->accounts->firstWhere('id', $this->selectedAccount);

        if (!$user->kyc_verified) {
            $this->addError('selectedAccount', 'Complete KYC before initiating a loan.');
            return;
        }

        if (!$account) {
            $this->addError('selectedAccount', 'Invalid account selected.');
            return;
        }

        if (\App\Models\Loan::where('idempotency_key', $this->idempotencyKey)->exists()) {
            $this->addError('selectedAccount', 'This loan has already been submitted.');
            return;
        }

        // -------------------------------
        // Amount cleanup (string)
        // -------------------------------
        $amount = preg_replace('/[^\d.]/', '', $this->amount);
        if (bccomp($amount, '0', 2) <= 0) {
            $this->addError('amount', 'Invalid amount.');
            return;
        }

        $minAmount = (string) config('loan.min_amount', '300');
        if (bccomp($amount, $minAmount, 2) === -1) {
            $this->addError('amount', "Minimum loan amount is {$this->currencySymbol}{$minAmount}.");
            return;
        }

        // -------------------------------
        // Income eligibility
        // -------------------------------
        $netIncome = LoanCalculator::parseNetIncome($this->net_income); // MUST return string
        if (bccomp($netIncome, '0', 2) <= 0) {
            $this->addError('net_income', 'Invalid net income value.');
            return;
        }
        $incomeRanges = LoanOptions::netIncomeRanges(); // assign to variable
        $lastRange = end($incomeRanges); // now it works

        // Skip max loan check if user selected the last range (open-ended)
        if ($this->net_income !== $lastRange) {
            $maxLoan = LoanCalculator::maxLoan($netIncome); // string

            // Format nicely with commas and 2 decimals
            $formattedMaxLoan = 'Bs' . number_format((float) $maxLoan, 2);

            if (bccomp($amount, $maxLoan, 2) === 1) {
                $this->addError('amount', "Maximum allowed loan based on your income is {$formattedMaxLoan}.");
                return;
            }
        }

        // -------------------------------
        // Duration + rate
        // -------------------------------
        $duration = preg_replace('/[^\d]/', '', $this->duration);
        if (bccomp($duration, '0', 0) <= 0) {
            $this->addError('duration', 'Invalid loan duration.');
            return;
        }

        $rate = (string) config("loan.interest_rates.{$this->credit_facility}", '0.04');

        // -------------------------------
        // Generate repayment schedule
        // -------------------------------
        $schedule = $this->repaymentSchedule;

        if (empty($schedule)) {
            $this->addError('amount', 'Cannot generate repayment schedule.');
            return;
        }

        // Compute totals from schedule
        $totalPrincipal = '0';
        $totalInterest = '0';
        foreach ($schedule as $item) {
            $totalPrincipal = bcadd($totalPrincipal, $item['principal'], 2);
            $totalInterest = bcadd($totalInterest, $item['interest'], 2);
        }

        $totalRepayment = bcadd($totalPrincipal, $totalInterest, 2);

        // Take first month total as representative monthly repayment
        $monthlyRepayment = $schedule[0]['total'];

        // -------------------------------
        // Persist confirmation state
        // -------------------------------
        $this->confirmedAmount = $amount;
        $this->confirmedDuration = $duration;
        $this->confirmedRate = $rate;
        $this->confirmedTotalInterest = $totalInterest;
        $this->confirmedMonthlyRepayment = $monthlyRepayment;
        $this->confirmedAccountId = $account->id;

        // Open modal
        $this->dispatch('open-loan-confirmation');
    }

    public function applyLoan()
    {
        $user = auth()->user();
        $account = $this->accounts->firstWhere('id', $this->confirmedAccountId);

        if (!$account || bccomp($this->confirmedAmount, '0', 2) <= 0) {
            $this->addError('amount', 'Invalid loan confirmation state.');
            return;
        }

        try {
            DB::beginTransaction();

            $loan = \App\Models\Loan::create([
                'user_id' => $user->id,
                'idempotency_key' => $this->idempotencyKey,
                'account_id' => $account->id,
                'amount' => $this->confirmedAmount,
                'duration' => $this->confirmedDuration,
                'credit_facility' => $this->credit_facility,
                'purpose' => $this->purpose,
                'interest_rate' => $this->confirmedRate,
                'total_interest' => $this->confirmedTotalInterest,
                'monthly_repayment' => $this->confirmedMonthlyRepayment,
                'principal_outstanding' => $this->confirmedAmount, // full principal initially
                'interest_outstanding' => $this->confirmedTotalInterest, // full interest initially
                'status' => 'pending',
                'first_due_date' => now()->addMonth(),
                'last_due_date' => now()->addMonths((int) $this->confirmedDuration),
            ]);

            $transaction = Transaction::create([
                'user_id' => $user->id,
                'idempotency_key' => $this->idempotencyKey,
                'account_id' => $account->id,
                'loan_id' => $loan->id,
                'type' => 'loan',
                'amount' => $this->confirmedAmount,
                'status' => Transaction::STATUS_PENDING,
                'balance_after' => $account->available_balance,
                'description' => 'Loan disbursement',
            ]);

            $user->notify(new TransactionNotification($transaction));

            AdminNotifier::notify(
                'Loan Request Submitted',
                [
                    'loan_id' => $loan->id,
                    'account_id' => $account->id,
                    'amount' => $transaction->amount,
                ],
                $user,
            );

            DB::commit();

            // Reset idempotency AFTER success
            $this->idempotencyKey = (string) Str::uuid();
            $this->dispatch('loan-success');
            $this->dispatch('transaction-success', type: 'loan', amount: $this->confirmedAmount);
            $this->dispatch('notification-sent');
            $this->reset(['amount', 'duration', 'credit_facility', 'employment_status', 'net_income', 'purpose', 'confirmedAmount', 'confirmedDuration', 'confirmedRate', 'confirmedTotalInterest', 'confirmedMonthlyRepayment', 'confirmedAccountId']);
        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Loan application failed', ['error' => $e]);
            $this->addError('amount', 'Something went wrong while processing your loan.');
        }
    }
};
?>

 <?php $__env->slot('title', null, []); ?> 
    <?php echo e(config('app.name')); ?> || Loan Application
 <?php $__env->endSlot(); ?>

<div class="space-y-5 py-4 font-sans">
    <div class="flex items-center justify-between gap-2 flex-wrap">
        <a href="<?php echo e(route('user.loans')); ?>"
            class="bg-sky-600 text-white flex items-center hover:shadow-2xl justify-center gap-1 px-3 py-2 rounded-md">
            <?php if (isset($component)) { $__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.arrow-left','data' => ['variant' => 'solid','class' => 'w-4 h-4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.arrow-left'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-4 h-4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08)): ?>
<?php $attributes = $__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08; ?>
<?php unset($__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08); ?>
<?php endif; ?>
<?php if (isset($__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08)): ?>
<?php $component = $__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08; ?>
<?php unset($__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08); ?>
<?php endif; ?>
            <span class="text-sm font-medium">Back to Information</span>
        </a>

        <a href="<?php echo e(route('user.loans.index')); ?>"
            class="bg-sky-600 text-white flex items-center hover:shadow-2xl justify-center gap-1 px-3 py-2 rounded-md">
            <?php if (isset($component)) { $__componentOriginal8e08e0a2beb8b59f88463fb57bbae428 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal8e08e0a2beb8b59f88463fb57bbae428 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.hand-coins','data' => ['class' => 'w-4 h-4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.hand-coins'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-4 h-4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal8e08e0a2beb8b59f88463fb57bbae428)): ?>
<?php $attributes = $__attributesOriginal8e08e0a2beb8b59f88463fb57bbae428; ?>
<?php unset($__attributesOriginal8e08e0a2beb8b59f88463fb57bbae428); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal8e08e0a2beb8b59f88463fb57bbae428)): ?>
<?php $component = $__componentOriginal8e08e0a2beb8b59f88463fb57bbae428; ?>
<?php unset($__componentOriginal8e08e0a2beb8b59f88463fb57bbae428); ?>
<?php endif; ?>
            <span class="text-sm font-medium">View Loans History</span>
        </a>
    </div>

    <div class="bg-white dark:bg-gray-900 border border-gray-100 dark:border-gray-800 rounded-xl p-4 md:p-6 space-y-4">
        <h2 class="flex items-center gap-3 text-sm md:text-base font-semibold text-gray-800 dark:text-gray-100">
            <span
                class="bg-sky-500/10 dark:bg-sky-400/10 w-9 h-9 flex items-center justify-center rounded-full text-sky-600 dark:text-sky-400">
                <?php if (isset($component)) { $__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.banknotes','data' => ['variant' => 'solid','class' => 'w-5 h-5']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.banknotes'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-5 h-5']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4)): ?>
<?php $attributes = $__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4; ?>
<?php unset($__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4)): ?>
<?php $component = $__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4; ?>
<?php unset($__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4); ?>
<?php endif; ?>
            </span>
            Loan Details
        </h2>

        <form wire:submit.prevent="confirmLoan" class="space-y-6 p-4 bg-white dark:bg-gray-800 rounded-lg shadow-sm">
            <div class="flex items-center justify-between gap-3 flex-wrap">
                <h3 class="text-lg font-bold text-gray-900 dark:text-gray-100 mb-4">
                    Loan Application
                </h3>
                <span class="text-red-400 text-sm font-medium capitalize">* all fields are required</span>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>

                    

                    
                    <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.defer' => 'selectedAccount','label' => 'Select Account','placeholder' => 'Select Account You\'re Applying Loan On']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'selectedAccount','label' => 'Select Account','placeholder' => 'Select Account You\'re Applying Loan On']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $accounts; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $account): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <option value="<?php echo e($account->id); ?>">
                                <?php echo e($account->bank_account_type); ?> - <?php echo e($account->masked_account_number); ?> -
                                <?php echo e($currencySymbol); ?><?php echo e($account->available_balance); ?>

                            </option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                </div>
                
                <div>
                    <?php if (isset($component)) { $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.input','data' => ['wire:model.defer' => 'amount','label' => 'Loan Amount','type' => 'number','step' => '0.01','placeholder' => 'Enter desired loan amount','required' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'amount','label' => 'Loan Amount','type' => 'number','step' => '0.01','placeholder' => 'Enter desired loan amount','required' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $attributes = $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $component = $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
                </div>

                
                <div>
                    <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.defer' => 'duration','label' => 'Loan Duration','placeholder' => 'Select duration','required' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'duration','label' => 'Loan Duration','placeholder' => 'Select duration','required' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->durations; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $d): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <option value="<?php echo e($d); ?>"><?php echo e($d); ?> Months</option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                </div>

                
                <div>
                    <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.defer' => 'credit_facility','label' => 'Credit Facility','placeholder' => 'Select type','required' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'credit_facility','label' => 'Credit Facility','placeholder' => 'Select type','required' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->creditFacilities; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $cf): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <option value="<?php echo e($cf); ?>"><?php echo e(ucfirst($cf)); ?> Loan</option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                </div>

                
                <div class="md:col-span-2 mt-4">
                    <h4 class="flex items-center gap-3 text-base font-semibold text-gray-800 dark:text-gray-100 mb-2">
                        <span
                            class="bg-sky-500/10 dark:bg-sky-400/10 w-9 h-9 flex items-center justify-center rounded-full text-sky-600 dark:text-sky-400">
                            <?php if (isset($component)) { $__componentOriginaldcfb31f33d929c53349fcb6cda5c8cbd = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaldcfb31f33d929c53349fcb6cda5c8cbd = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.wallet','data' => ['variant' => 'solid','class' => 'w-5 h-5']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.wallet'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-5 h-5']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaldcfb31f33d929c53349fcb6cda5c8cbd)): ?>
<?php $attributes = $__attributesOriginaldcfb31f33d929c53349fcb6cda5c8cbd; ?>
<?php unset($__attributesOriginaldcfb31f33d929c53349fcb6cda5c8cbd); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaldcfb31f33d929c53349fcb6cda5c8cbd)): ?>
<?php $component = $__componentOriginaldcfb31f33d929c53349fcb6cda5c8cbd; ?>
<?php unset($__componentOriginaldcfb31f33d929c53349fcb6cda5c8cbd); ?>
<?php endif; ?>
                        </span>
                        Financial Information
                    </h4>
                </div>

                
                <div>
                    <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.defer' => 'net_income','label' => 'Monthly Net Income','placeholder' => 'Select range','required' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'net_income','label' => 'Monthly Net Income','placeholder' => 'Select range','required' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->netIncomeRanges; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $range): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <option value="<?php echo e($range); ?>"><?php echo e($range); ?></option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                </div>

                
                <div>
                    <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.defer' => 'employment_status','label' => 'Employment Status','placeholder' => 'Select status','required' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'employment_status','label' => 'Employment Status','placeholder' => 'Select status','required' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->employmentStatuses; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $es): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                            <option value="<?php echo e($es); ?>"><?php echo e(ucwords(str_replace('-', ' ', $es))); ?></option>
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
                </div>

                
                <div class="md:col-span-2">
                    <?php if (isset($component)) { $__componentOriginal4727f9fd7c3055c2cf9c658d89b16886 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.textarea','data' => ['wire:model.defer' => 'purpose','label' => 'Loan Purpose','placeholder' => 'Briefly describe the purpose of the loan','rows' => '3','required' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('textarea'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'purpose','label' => 'Loan Purpose','placeholder' => 'Briefly describe the purpose of the loan','rows' => '3','required' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886)): ?>
<?php $attributes = $__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886; ?>
<?php unset($__attributesOriginal4727f9fd7c3055c2cf9c658d89b16886); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal4727f9fd7c3055c2cf9c658d89b16886)): ?>
<?php $component = $__componentOriginal4727f9fd7c3055c2cf9c658d89b16886; ?>
<?php unset($__componentOriginal4727f9fd7c3055c2cf9c658d89b16886); ?>
<?php endif; ?>
                </div>

                
                <div
                    class="md:col-span-2 mt-4 space-y-3 p-4 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                    <h4 class="text-gray-800 dark:text-gray-100 font-semibold mb-2">Agreements & Consent</h4>

                    
                    <div class="flex items-start gap-2">
                        <input type="checkbox" wire:model.defer="agreement_terms" id="agreement_terms" required
                            class="mt-1 rounded border-gray-300 text-sky-600 focus:ring-sky-500">
                        <label for="agreement_terms" class="text-gray-700 dark:text-gray-200 text-sm">
                            I agree to the <a href="#" class="text-sky-600 hover:underline">Terms &
                                Conditions</a>.
                        </label>
                    </div>

                    
                    <div class="flex items-start gap-2">
                        <input type="checkbox" wire:model.defer="agreement_privacy" id="agreement_privacy" required
                            class="mt-1 rounded border-gray-300 text-sky-600 focus:ring-sky-500">
                        <label for="agreement_privacy" class="text-gray-700 dark:text-gray-200 text-sm">
                            I acknowledge the <a href="#" class="text-sky-600 hover:underline">Privacy Policy</a>.
                        </label>
                    </div>

                    
                    <div class="flex items-start gap-2">
                        <input type="checkbox" wire:model.defer="agreement_credit" id="agreement_credit" required
                            class="mt-1 rounded border-gray-300 text-sky-600 focus:ring-sky-500">
                        <label for="agreement_credit" class="text-gray-700 dark:text-gray-200 text-sm">
                            I authorize <?php echo e(config('app.name')); ?> to verify my information and credit history.
                        </label>
                    </div>

                    
                    <p class="text-gray-600 dark:text-gray-300 text-sm mt-2">
                        By submitting this application, I confirm that all information provided is accurate and
                        complete.
                    </p>
                </div>

                
                <div class="md:col-span-2 flex items-center gap-4 mt-4">
                    <div class="flex items-center gap-2 text-sm text-gray-500 dark:text-gray-400">
                        <?php if (isset($component)) { $__componentOriginalf870514c33bb1b53395ba02235f60146 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalf870514c33bb1b53395ba02235f60146 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.shield-check','data' => ['class' => 'w-5 h-5 text-green-500']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.shield-check'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-5 h-5 text-green-500']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalf870514c33bb1b53395ba02235f60146)): ?>
<?php $attributes = $__attributesOriginalf870514c33bb1b53395ba02235f60146; ?>
<?php unset($__attributesOriginalf870514c33bb1b53395ba02235f60146); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalf870514c33bb1b53395ba02235f60146)): ?>
<?php $component = $__componentOriginalf870514c33bb1b53395ba02235f60146; ?>
<?php unset($__componentOriginalf870514c33bb1b53395ba02235f60146); ?>
<?php endif; ?> Secure & Encrypted
                    </div>
                    <div class="flex items-center gap-2 text-sm text-gray-500 dark:text-gray-400">
                        <?php if (isset($component)) { $__componentOriginal7649f9fde3f65e39f506d39dd1ac88cb = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal7649f9fde3f65e39f506d39dd1ac88cb = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.lock-closed','data' => ['class' => 'w-5 h-5 text-blue-500']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.lock-closed'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-5 h-5 text-blue-500']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal7649f9fde3f65e39f506d39dd1ac88cb)): ?>
<?php $attributes = $__attributesOriginal7649f9fde3f65e39f506d39dd1ac88cb; ?>
<?php unset($__attributesOriginal7649f9fde3f65e39f506d39dd1ac88cb); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal7649f9fde3f65e39f506d39dd1ac88cb)): ?>
<?php $component = $__componentOriginal7649f9fde3f65e39f506d39dd1ac88cb; ?>
<?php unset($__componentOriginal7649f9fde3f65e39f506d39dd1ac88cb); ?>
<?php endif; ?> Trusted Platform
                    </div>
                    <div class="flex items-center gap-2 text-sm text-gray-500 dark:text-gray-400">
                        <?php if (isset($component)) { $__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.credit-card','data' => ['class' => 'w-5 h-5 text-yellow-500']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.credit-card'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-5 h-5 text-yellow-500']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146)): ?>
<?php $attributes = $__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146; ?>
<?php unset($__attributesOriginal6e0b21ef9231e6606d7ac9c0c02dc146); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146)): ?>
<?php $component = $__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146; ?>
<?php unset($__componentOriginal6e0b21ef9231e6606d7ac9c0c02dc146); ?>
<?php endif; ?> Protected Transactions
                    </div>
                </div>
            </div>

            
            <div class="mt-6">
                <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['type' => 'submit','class' => 'w-full']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['type' => 'submit','class' => 'w-full']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                    Apply for Loan
                 <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
            </div>
        </form>

    </div>

    <!-- CONFIRMATION MODAL -->
    <!-- CONFIRMATION MODAL -->
    <?php if (isset($component)) { $__componentOriginal9f64f32e90b9102968f2bc548315018c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal9f64f32e90b9102968f2bc548315018c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.modal','data' => ['maxWidth' => 'max-w-lg','@openLoanConfirmation.window' => 'openModal()','@closeLoanConfirmation.window' => 'close()','xOn:loanSuccess.window' => 'close()']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('modal'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['maxWidth' => 'max-w-lg','@open-loan-confirmation.window' => 'openModal()','@close-loan-confirmation.window' => 'close()','x-on:loan-success.window' => 'close()']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>


        <div class="space-y-6">

            <!-- Header -->
            <div class="text-center space-y-3">
                <div
                    class="w-14 h-14 bg-linear-to-br from-sky-500 to-sky-800 rounded-2xl mx-auto flex items-center justify-center text-white shadow-lg">
                    <?php if (isset($component)) { $__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.banknotes','data' => ['variant' => 'solid','class' => 'w-6 h-6']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.banknotes'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-6 h-6']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4)): ?>
<?php $attributes = $__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4; ?>
<?php unset($__attributesOriginal1a2aab62646bbf4070a26cfe0540f0d4); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4)): ?>
<?php $component = $__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4; ?>
<?php unset($__componentOriginal1a2aab62646bbf4070a26cfe0540f0d4); ?>
<?php endif; ?>
                </div>

                <h2 class="text-lg font-bold text-gray-900 dark:text-gray-100">
                    Confirm Loan Application
                </h2>

                <p class="text-sm text-gray-500 dark:text-gray-300 max-w-xs mx-auto">
                    Please review your loan details carefully before submitting.
                </p>
            </div>

            <!-- Summary -->
            <?php
                // Compute totals from the repayment schedule
                $totalPrincipal = 0;
                $totalInterest = 0;
                foreach ($this->repaymentSchedule as $item) {
                    $totalPrincipal += (float) $item['principal'];
                    $totalInterest += (float) $item['interest'];
                }
                $totalRepayment = $totalPrincipal + $totalInterest;
            ?>

            <div class="bg-gray-50 dark:bg-gray-800 rounded-xl p-4 border border-gray-200 dark:border-gray-700">
                <ul class="space-y-2 text-sm text-gray-700 dark:text-gray-200">

                    <li class="flex justify-between">
                        <span class="font-medium">Account</span>
                        <span><?php echo e($accounts->firstWhere('id', $selectedAccount)?->masked_account_number ?? '-'); ?></span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Loan Amount</span>
                        <span><?php echo e($currencySymbol); ?><?php echo e(number_format((float) $confirmedAmount, 2)); ?></span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Duration</span>
                        <span><?php echo e($confirmedDuration); ?> months</span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Credit Facility</span>
                        <span><?php echo e(ucfirst($credit_facility)); ?> Loan</span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Monthly Repayment</span>
                        <span class="font-semibold text-sky-600 dark:text-sky-400">
                            <?php echo e($currencySymbol); ?><?php echo e(number_format((float) $confirmedMonthlyRepayment, 2)); ?>

                        </span>
                    </li>

                    <li class="flex justify-between">
                        <span class="font-medium">Total Interest</span>
                        <span><?php echo e($currencySymbol); ?><?php echo e(number_format($totalInterest, 2)); ?></span>
                    </li>

                    <li class="flex justify-between border-t pt-2 mt-2">
                        <span class="font-semibold">Total Repayment</span>
                        <span class="font-bold">
                            <?php echo e($currencySymbol); ?><?php echo e(number_format($totalRepayment, 2)); ?>

                        </span>
                    </li>
                </ul>
            </div>

            <!-- Repayment Schedule -->
            <div class="space-y-2">
                <h4 class="font-semibold text-gray-800 dark:text-gray-100">
                    Repayment Schedule
                </h4>

                <div class="overflow-hidden rounded-lg border border-gray-200 dark:border-gray-700">
                    <table class="w-full text-sm">
                        <thead class="bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                            <tr>
                                <th class="px-3 py-2 text-left">Month</th>
                                <th class="px-3 py-2 text-right">Principal</th>
                                <th class="px-3 py-2 text-right">Interest</th>
                                <th class="px-3 py-2 text-right">Total</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white dark:bg-gray-800">
                            <?php
                                $runningPrincipal = 0;
                                $runningInterest = 0;
                            ?>
                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->repaymentSchedule; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $item): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
                                <?php
                                    $principal = number_format((float) $item['principal'], 2);
                                    $interest = number_format((float) $item['interest'], 2);
                                    $total = number_format((float) $item['total'], 2);

                                    $runningPrincipal += (float) $item['principal'];
                                    $runningInterest += (float) $item['interest'];
                                ?>

                                <tr class="border-t border-gray-200 dark:border-gray-700">
                                    <td class="px-3 py-2"><?php echo e($item['month']); ?></td>
                                    <td class="px-3 py-2 text-right"><?php echo e($currencySymbol); ?><?php echo e($principal); ?></td>
                                    <td class="px-3 py-2 text-right"><?php echo e($currencySymbol); ?><?php echo e($interest); ?></td>
                                    <td class="px-3 py-2 text-right font-semibold">
                                        <?php echo e($currencySymbol); ?><?php echo e($total); ?></td>
                                </tr>
                            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>

                            <!-- Totals Row -->
                            <tr
                                class="border-t border-gray-200 dark:border-gray-700 font-bold bg-gray-50 dark:bg-gray-700">
                                <td class="px-3 py-2 text-left">Total</td>
                                <td class="px-3 py-2 text-right">
                                    <?php echo e($currencySymbol); ?><?php echo e(number_format($runningPrincipal, 2)); ?></td>
                                <td class="px-3 py-2 text-right">
                                    <?php echo e($currencySymbol); ?><?php echo e(number_format($runningInterest, 2)); ?></td>
                                <td class="px-3 py-2 text-right">
                                    <?php echo e($currencySymbol); ?><?php echo e(number_format($runningPrincipal + $runningInterest, 2)); ?>

                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Actions -->
            <div class="space-y-3">
                <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['wire:click' => 'applyLoan','variant' => 'primary','class' => 'w-full bg-sky-600 hover:bg-sky-700']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'applyLoan','variant' => 'primary','class' => 'w-full bg-sky-600 hover:bg-sky-700']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                    Confirm & Submit
                 <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>

                <button type="button" @click="$dispatch('close-loan-confirmation')"
                    class="w-full text-center text-sm text-red-600 hover:underline">
                    Cancel
                </button>
            </div>
        </div>
     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal9f64f32e90b9102968f2bc548315018c)): ?>
<?php $attributes = $__attributesOriginal9f64f32e90b9102968f2bc548315018c; ?>
<?php unset($__attributesOriginal9f64f32e90b9102968f2bc548315018c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal9f64f32e90b9102968f2bc548315018c)): ?>
<?php $component = $__componentOriginal9f64f32e90b9102968f2bc548315018c; ?>
<?php unset($__componentOriginal9f64f32e90b9102968f2bc548315018c); ?>
<?php endif; ?>

</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/pages/user/⚡loans.blade.php ENDPATH**/ ?>