<?php

use Livewire\Component;
use Livewire\Attributes\Validate;
use App\Models\Grant;
use App\Models\GrantLog;
use Illuminate\Support\Facades\DB;
use App\Notifications\GrantConfirmationNotification;

new class extends Component {
    #[Validate('required', message: 'Please provide an amount')]
    #[Validate('numeric', message: 'Amount must be a number')]
    #[Validate('min:1', message: 'Enter a valid amount')]
    public $amount;

    #[
        Validate(
            [
                'subscription' => 'required|array|min:1',
                'subscription.*' => 'required|string|in:program_funding,equipment_funding,research_funding,community_outreach',
            ],
            message: [
                'subscription.required' => 'Please select at least one funding purpose.',
                'subscription.min' => 'Please select at least one funding purpose.',
                'subscription.*.in' => 'Invalid selection.',
            ],
        ),
    ]
    public $subscription = [];

    public $type = 'personal';

    public function mount()
    {
        $user = auth()->user();

        $existingGrant = Grant::where('user_id', $user->id)->where('type', 'personal')->where('status', 'pending')->first();

        if ($existingGrant) {
            return redirect()
                ->route('user.grants.index')
                ->with([
                    'message' => 'you have a pending grant request.',
                    'type' => 'info',
                ]);
        }
    }

    public function personalGrant()
    {
        $this->validate();
        $user = auth()->user();

        // Step 0: check KYC
        if (!$user->kyc_verified) {
            $this->addError('error', 'Complete KYC before submitting.');
            return;
        }

        try {
            DB::beginTransaction();

            $existingGrant = Grant::where('user_id', $user->id)
                ->where('type', 'personal')
                ->where('status', 'pending')
                ->lockForUpdate() // lock row to prevent race condition
                ->first();

            if ($existingGrant) {
                throw new \Exception('You already have a pending personal grant application.');
            }

            $grant = Grant::create([
                'type' => 'personal',
                'user_id' => auth()->id(),
                'amount' => $this->amount,
                'purposes' => $this->subscription,
                'status' => 'pending',
            ]);

            GrantLog::create([
                'grant_id' => $grant->id,
                'user_id' => auth()->id(),
                'action' => 'submitted',
                'data' => [
                    'amount' => $this->amount,
                    'purposes' => $this->subscription,
                ],
            ]);

            auth()->user()->notify(new GrantConfirmationNotification($grant));

            DB::commit();

            $this->dispatch('showToast', message: 'Your grant application has been submitted!', type: 'success');
            $this->dispatch('notification-sent');
            $this->dispatch('transaction-success', type: 'grant', amount: $this->amount);
            $this->reset(['amount', 'subscription']);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Grant submission failed: ' . $e->getMessage());
            $this->dispatch('showToast', message: 'There was an error submitting your grant. Please try again.', type: 'error');
        }
    }
};

?>

 <?php $__env->slot('title', null, []); ?> 
    <?php echo e(config('app.name')); ?> || Personal Grant Application
 <?php $__env->endSlot(); ?>


<div class="space-y-6 py-4 font-sans">

    <!-- Header / Navigation -->
    <div class="flex flex-wrap items-center justify-between gap-2">
        <a href="<?php echo e(route('user.grants.index')); ?>"
            class="flex items-center gap-1 px-3 py-1.5 rounded-md text-black dark:text-white hover:shadow hover:bg-gray-100 dark:hover:bg-gray-700 transition">
            <?php if (isset($component)) { $__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.arrow-left','data' => ['variant' => 'solid','class' => 'w-4 h-4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.arrow-left'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-4 h-4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08)): ?>
<?php $attributes = $__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08; ?>
<?php unset($__attributesOriginale41799ef1b2ae0b0e320ff9b21dacd08); ?>
<?php endif; ?>
<?php if (isset($__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08)): ?>
<?php $component = $__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08; ?>
<?php unset($__componentOriginale41799ef1b2ae0b0e320ff9b21dacd08); ?>
<?php endif; ?>
            <span class="text-sm font-medium">Individual Grant Application</span>
        </a>

        <a href="<?php echo e(route('user.grants.history')); ?>"
            class="flex items-center gap-1 px-3 py-1.5 rounded-md text-black dark:text-white hover:shadow hover:bg-gray-100 dark:hover:bg-gray-700 transition">
            <?php if (isset($component)) { $__componentOriginalc459039abf56f83b43258829ddb861ca = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc459039abf56f83b43258829ddb861ca = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.receipt-percent','data' => ['variant' => 'solid','class' => 'w-4 h-4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.receipt-percent'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-4 h-4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc459039abf56f83b43258829ddb861ca)): ?>
<?php $attributes = $__attributesOriginalc459039abf56f83b43258829ddb861ca; ?>
<?php unset($__attributesOriginalc459039abf56f83b43258829ddb861ca); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc459039abf56f83b43258829ddb861ca)): ?>
<?php $component = $__componentOriginalc459039abf56f83b43258829ddb861ca; ?>
<?php unset($__componentOriginalc459039abf56f83b43258829ddb861ca); ?>
<?php endif; ?>
            <span class="text-sm font-medium">Track Request</span>
        </a>

    </div>

    <!-- Card Wrapper -->
    <div
        class="bg-white/80 dark:bg-gray-800/80 backdrop-blur-md border border-gray-200/50 dark:border-gray-700/50 
               rounded-2xl shadow-md dark:shadow-lg p-6 sm:p-8 space-y-4">

        <!-- Header Icon + Title -->
        <div class="flex items-center gap-2">
            <?php if (isset($component)) { $__componentOriginalcbe89caa4ae8c58f7efd0ed6343c35ff = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalcbe89caa4ae8c58f7efd0ed6343c35ff = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.user','data' => ['variant' => 'solid','class' => 'w-4 h-4']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.user'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['variant' => 'solid','class' => 'w-4 h-4']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalcbe89caa4ae8c58f7efd0ed6343c35ff)): ?>
<?php $attributes = $__attributesOriginalcbe89caa4ae8c58f7efd0ed6343c35ff; ?>
<?php unset($__attributesOriginalcbe89caa4ae8c58f7efd0ed6343c35ff); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalcbe89caa4ae8c58f7efd0ed6343c35ff)): ?>
<?php $component = $__componentOriginalcbe89caa4ae8c58f7efd0ed6343c35ff; ?>
<?php unset($__componentOriginalcbe89caa4ae8c58f7efd0ed6343c35ff); ?>
<?php endif; ?>
            <h2 class="text-sm md:text-base capitalize font-semibold text-gray-900 dark:text-white">Apply as Individual
            </h2>

        </div>

        <form wire:submit.prevent='personalGrant' class="space-y-4">
            <div>
                <?php if (isset($component)) { $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.input','data' => ['label' => 'Requested Amount *','type' => 'number','step' => '0.01','min' => '1','placeholder' => 'Enter the amount you would like to request for your grant','wire:model.live.blur' => 'amount']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['label' => 'Requested Amount *','type' => 'number','step' => '0.01','min' => '1','placeholder' => 'Enter the amount you would like to request for your grant','wire:model.live.blur' => 'amount']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $attributes = $__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__attributesOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1)): ?>
<?php $component = $__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1; ?>
<?php unset($__componentOriginalc2fcfa88dc54fee60e0757a7e0572df1); ?>
<?php endif; ?>
            </div>

            <?php
                $fundingOptions = [
                    [
                        'value' => 'program_funding',
                        'title' => 'Program Funding',
                        'description' => 'Educational, cultural, or social programs',
                        'icon' => 'academic-cap',
                    ],
                    [
                        'value' => 'equipment_funding',
                        'title' => 'Equipment Funding',
                        'description' => 'Equipment or technology purchases',
                        'icon' => 'cpu-chip',
                    ],
                    [
                        'value' => 'research_funding',
                        'title' => 'Research Funding',
                        'description' => 'Research or field studies',
                        'icon' => 'beaker',
                    ],
                    [
                        'value' => 'community_outreach',
                        'title' => 'Community Outreach',
                        'description' => 'Community and underserved population support',
                        'icon' => 'users',
                    ],
                ];

            ?>
            <?php if (isset($component)) { $__componentOriginal6fe78fe0f4ef07e907edf153e0b8612e = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal6fe78fe0f4ef07e907edf153e0b8612e = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.checkbox-card-group','data' => ['label' => 'Please select all funding purposes that apply to your application:','wire:model.live.blur' => 'subscription','options' => $fundingOptions,'columns' => '2','error' => 'subscription']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('checkbox-card-group'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['label' => 'Please select all funding purposes that apply to your application:','wire:model.live.blur' => 'subscription','options' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute($fundingOptions),'columns' => '2','error' => 'subscription']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal6fe78fe0f4ef07e907edf153e0b8612e)): ?>
<?php $attributes = $__attributesOriginal6fe78fe0f4ef07e907edf153e0b8612e; ?>
<?php unset($__attributesOriginal6fe78fe0f4ef07e907edf153e0b8612e); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal6fe78fe0f4ef07e907edf153e0b8612e)): ?>
<?php $component = $__componentOriginal6fe78fe0f4ef07e907edf153e0b8612e; ?>
<?php unset($__componentOriginal6fe78fe0f4ef07e907edf153e0b8612e); ?>
<?php endif; ?>

            <div
                class="bg-gray-100 dark:bg-gray-700 p-4 rounded-lg flex flex-col sm:flex-row gap-3 items-start sm:items-center shadow-sm">
                <?php if (isset($component)) { $__componentOriginal1f8061448e375a811323d4736f7bf58b = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal1f8061448e375a811323d4736f7bf58b = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.information-circle','data' => ['class' => 'w-5 h-5 text-sky-500 shrink-0']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.information-circle'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'w-5 h-5 text-sky-500 shrink-0']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal1f8061448e375a811323d4736f7bf58b)): ?>
<?php $attributes = $__attributesOriginal1f8061448e375a811323d4736f7bf58b; ?>
<?php unset($__attributesOriginal1f8061448e375a811323d4736f7bf58b); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal1f8061448e375a811323d4736f7bf58b)): ?>
<?php $component = $__componentOriginal1f8061448e375a811323d4736f7bf58b; ?>
<?php unset($__componentOriginal1f8061448e375a811323d4736f7bf58b); ?>
<?php endif; ?>
                <div class="space-y-1">
                    <h3 class="font-semibold text-sm tracking-wide">Important Information
                    </h3>
                    <p class="text-xs text-gray-600 dark:text-sky-200 leading-relaxed">
                        By submitting this application, you acknowledge that the final approved amount will be
                        determined during our review process based on your eligibility and requested amount. You'll
                        receive notification once your application has been processed.
                    </p>
                </div>
            </div>

            <?php if (isset($component)) { $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.button','data' => ['type' => 'submit','size' => 'md','icon' => 'send','iconPosition' => 'right']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['type' => 'submit','size' => 'md','icon' => 'send','iconPosition' => 'right']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                Submit Application
             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $attributes = $__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__attributesOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>
<?php if (isset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561)): ?>
<?php $component = $__componentOriginald0f1fd2689e4bb7060122a5b91fe8561; ?>
<?php unset($__componentOriginald0f1fd2689e4bb7060122a5b91fe8561); ?>
<?php endif; ?>


        </form>
    </div>

</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/pages/user/⚡personal-grant.blade.php ENDPATH**/ ?>