<?php

use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Traits\WithTableHelpers;
use Illuminate\Validation\Rule;
use Livewire\Attributes\Computed;
use Illuminate\Validation\Rules\Password;
use Livewire\WithPagination;
use App\Models\Admin;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Hash;

new #[Layout('layouts::base', ['title' => 'Admin Setting'])] class extends Component {
    use WithPagination, WithTableHelpers, WithFileUploads;

    public $name, $email, $password, $password_confirmation;
    public $editingAdmin = null;

    public int $perPage = 10;
    public int $page = 1;
    public $avatar;
    public $role;
    public $roles = ['super_admin', 'admin', 'manager']; // define roles

    /**
     * Fields that can be searched
     */
    protected function searchableFields(): array
    {
        return ['id', 'name', 'email'];
    }

    /**
     * Validation rules
     */
    protected function rules()
    {
        $rules = [
            'name' => 'required|string|max:255|unique:admins,name',
            'email' => ['required', 'email', 'max:255', Rule::unique(Admin::class, 'email')],
            'avatar' => $this->editingAdmin ? 'nullable|image|mimes:jpg,webp,avif,jpeg,png,svg|max:2048' : 'required|image|mimes:jpg,webp,avif,jpeg,png,svg|max:2048',
            'password' => ['required', 'confirmed', Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised()],
            'role' => 'required|string|in:' . implode(',', $this->roles),
        ];

        if ($this->editingAdmin) {
            $rules['name'] = ['required', 'string', 'max:255', Rule::unique('admins', 'name')->ignore($this->editingAdmin)];
            $rules['email'] = ['required', 'email', 'max:255', Rule::unique('admins', 'email')->ignore($this->editingAdmin)];

            // Password optional on edit
            $rules['password'] = ['nullable', 'confirmed', Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised()];
        }

        return $rules;
    }

    /**
     * Validate only a single property on update
     */
    public function updated($property)
    {
        $this->validateOnly($property);
    }

    /**
     * Reset pagination when perPage changes
     */
    public function updatedPerPage(): void
    {
        $this->resetPage();
        $this->dispatch('perPageUpdated', $this->perPage);
    }

    /**
     * Delete an admin
     */
    public function delete($id)
    {
        $adminToDelete = Admin::findOrFail($id);
        $currentAdmin = auth('admin')->user(); // use admin guard

        // Prevent deleting last admin
        if (Admin::count() <= 1) {
            $this->dispatch('showToast', 'Cannot delete the last remaining admin!', type: 'error');
            return;
        }

        // Prevent deleting self
        if ($adminToDelete->id === $currentAdmin->id) {
            $this->dispatch('showToast', 'You cannot delete your own account!', type: 'error');
            return;
        }

        // Prevent deleting super admin if logged-in admin is not super admin
        if ($adminToDelete->role === 'super_admin' && $currentAdmin->role !== 'super_admin') {
            $this->dispatch('showToast', 'You cannot delete a Super Admin!', type: 'error');
            return;
        }

        if ($adminToDelete->avatar) {
            \Storage::disk('public')->delete($adminToDelete->avatar);
        }

        $adminToDelete->delete();

        $this->dispatch('showToast', 'Admin successfully deleted', type: 'success');
    }

    /**
     * Reset the form
     */
    public function resetForm()
    {
        $this->reset(['name', 'email', 'password', 'password_confirmation', 'avatar', 'editingAdmin']);

        // Set default role
        $this->role = auth('admin')->user()->role === 'super_admin' ? 'admin' : 'manager';

        $this->resetValidation();
    }
    /**
     * Populate form for editing
     */
    public function editAdmin($id)
    {
        $admin = Admin::findOrFail($id);
        $this->editingAdmin = $admin->id;
        $this->name = $admin->name;
        $this->email = $admin->email;
        $this->role = $admin->role;
        $this->password = '';
        $this->password_confirmation = '';
    }

    /**
     * Save a new or edited admin
     */
    public function saveAdmin()
    {
        $currentUser = auth('admin')->user();

        // Determine allowed roles based on logged-in admin
        $allowedRoles = $currentUser->role === 'super_admin' ? $this->roles : array_values(array_diff($this->roles, ['super_admin']));

        // If editing, allow the current admin's role even if restricted
        if ($this->editingAdmin) {
            $editingAdminRole = Admin::find($this->editingAdmin)->role;
            if (!in_array($editingAdminRole, $allowedRoles)) {
                $allowedRoles[] = $editingAdminRole;
            }
        }

        // Base validation rules
        $rules = [
            'name' => ['required', 'string', 'max:255', $this->editingAdmin ? Rule::unique('admins', 'name')->ignore($this->editingAdmin) : Rule::unique('admins', 'name')],
            'email' => ['required', 'email', 'max:255', $this->editingAdmin ? Rule::unique('admins', 'email')->ignore($this->editingAdmin) : Rule::unique('admins', 'email')],
            'avatar' => $this->editingAdmin ? 'nullable|image|mimes:jpg,webp,avif,jpeg,png,svg|max:2048' : 'required|image|mimes:jpg,webp,avif,jpeg,png,svg|max:2048',
            'password' => $this->editingAdmin ? ['nullable', 'confirmed', Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised()] : ['required', 'confirmed', Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised()],
            'role' => ['required', 'string', Rule::in($allowedRoles)],
        ];

        // Validate
        $validated = $this->validate($rules);

        // Hash password if provided
        if (!empty($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        } else {
            unset($validated['password']); // keep old password
        }

        // Handle avatar upload
        if ($this->avatar) {
            $path = $this->avatar->store('admin-avatar', 'public');
            $validated['avatar'] = $path;

            if ($this->editingAdmin) {
                $admin = Admin::find($this->editingAdmin);
                if ($admin->avatar) {
                    \Storage::disk('public')->delete($admin->avatar);
                }
            }
        }

        // Save or update
        if ($this->editingAdmin) {
            Admin::findOrFail($this->editingAdmin)->update($validated);
            $message = 'Admin updated successfully!';
        } else {
            Admin::create($validated);
            $message = 'Admin created successfully!';
        }

        $this->dispatch('showToast', message: $message, type: 'success');
        $this->resetForm();
    }

    #[Computed]
    public function admins()
    {
        $query = Admin::query();

        $admin = auth('admin')->user(); // <-- use your admin guard

        if ($admin && $admin->role === 'super_admin') {
            // super admin sees all
        } else {
            // non-super admin cannot see super admins
            $query->where('role', '!=', 'super_admin');
        }

        $this->applySearch($query, $this->searchableFields());

        return $query
            ->orderBy($this->sortField ?? 'id', $this->sortDirection ?? 'desc')
            ->paginate($this->perPage)
            ->withQueryString();
    }

    public function clearError()
    {
        $this->resetErrorBag('avatar');
    }
    public function updatedAvatar()
    {
        $this->clearError();
    }
};
?>

<div class="min-h-screen bg-gray-50 dark:bg-gray-950 p-6 space-y-6">
    <!-- Dynamic Page Title -->
    <div class="mb-4">
        <h2 class="text-xl font-semibold text-gray-800 dark:text-gray-100">
            Manage Admin
        </h2>
    </div>
    
    <div class="mb-6 p-4 bg-gray-50 dark:bg-gray-800 rounded shadow">
        <h3 class="font-semibold text-lg mb-4">
            <?php echo e($editingAdmin ? 'Edit Admin' : 'Add New Admin'); ?>

        </h3>
        <div class="space-y-4">
     <?php if (isset($component)) { $__componentOriginaled2cde6083938c436304f332ba96bb7c = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginaled2cde6083938c436304f332ba96bb7c = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.select','data' => ['wire:model.defer' => 'role','label' => 'Role','options' => collect($roles)
        ->when(auth('admin')->user()->role !== 'super_admin', fn($q) => $q->reject(fn($r) => $r === 'super_admin'))
        ->mapWithKeys(fn($r) => [$r => ucfirst(str_replace('_', ' ', $r))])
        ->toArray(),'placeholder' => 'Select Role']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('select'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'role','label' => 'Role','options' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute(collect($roles)
        ->when(auth('admin')->user()->role !== 'super_admin', fn($q) => $q->reject(fn($r) => $r === 'super_admin'))
        ->mapWithKeys(fn($r) => [$r => ucfirst(str_replace('_', ' ', $r))])
        ->toArray()),'placeholder' => 'Select Role']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $attributes = $__attributesOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__attributesOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
<?php if (isset($__componentOriginaled2cde6083938c436304f332ba96bb7c)): ?>
<?php $component = $__componentOriginaled2cde6083938c436304f332ba96bb7c; ?>
<?php unset($__componentOriginaled2cde6083938c436304f332ba96bb7c); ?>
<?php endif; ?>
            <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'name','label' => 'Admin Name','placeholder' => 'admin']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'name','label' => 'Admin Name','placeholder' => 'admin']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
            <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'email','label' => 'Admin Email','placeholder' => 'admin@example.com']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'email','label' => 'Admin Email','placeholder' => 'admin@example.com']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
            <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'password','label' => __('Password'),'type' => 'password','autocomplete' => 'password','placeholder' => __('Password'),'viewable' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'password','label' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute(__('Password')),'type' => 'password','autocomplete' => 'password','placeholder' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute(__('Password')),'viewable' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>

            <?php if (isset($component)) { $__componentOriginal26c546557cdc09040c8dd00b2090afd0 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginal26c546557cdc09040c8dd00b2090afd0 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::input.index','data' => ['wire:model.defer' => 'password_confirmation','label' => __('Confirm Password'),'type' => 'password','autocomplete' => 'confirm-password','placeholder' => __('Confirm Password'),'viewable' => true]] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::input'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:model.defer' => 'password_confirmation','label' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute(__('Confirm Password')),'type' => 'password','autocomplete' => 'confirm-password','placeholder' => \Illuminate\View\Compilers\BladeCompiler::sanitizeComponentAttribute(__('Confirm Password')),'viewable' => true]); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $attributes = $__attributesOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__attributesOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
<?php if (isset($__componentOriginal26c546557cdc09040c8dd00b2090afd0)): ?>
<?php $component = $__componentOriginal26c546557cdc09040c8dd00b2090afd0; ?>
<?php unset($__componentOriginal26c546557cdc09040c8dd00b2090afd0); ?>
<?php endif; ?>
            <?php if (isset($component)) { $__componentOriginalf3cfdacf9d0c6803e044223ec6e3c447 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalf3cfdacf9d0c6803e044223ec6e3c447 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'components.ui.custom-file-upload','data' => ['model' => 'avatar','label' => 'Upload Admin Avatar','acceptedFileTypes' => 'image/*']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('ui.custom-file-upload'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['model' => 'avatar','label' => 'Upload Admin Avatar','accepted-file-types' => 'image/*']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalf3cfdacf9d0c6803e044223ec6e3c447)): ?>
<?php $attributes = $__attributesOriginalf3cfdacf9d0c6803e044223ec6e3c447; ?>
<?php unset($__attributesOriginalf3cfdacf9d0c6803e044223ec6e3c447); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalf3cfdacf9d0c6803e044223ec6e3c447)): ?>
<?php $component = $__componentOriginalf3cfdacf9d0c6803e044223ec6e3c447; ?>
<?php unset($__componentOriginalf3cfdacf9d0c6803e044223ec6e3c447); ?>
<?php endif; ?>
            <?php if (isset($component)) { $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::button.index','data' => ['wire:click' => 'saveAdmin','variant' => 'primary','color' => 'indigo','class' => 'w-full text-center py-3']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'saveAdmin','variant' => 'primary','color' => 'indigo','class' => 'w-full text-center py-3']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                <?php echo e($editingAdmin ? 'Update Admin' : 'Add Admin'); ?>

             <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $attributes = $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $component = $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
            <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($editingAdmin): ?>
                <button wire:click="resetForm"
                    class="mt-3 px-4 py-2 bg-gray-400 text-white rounded hover:bg-gray-500 ml-2">
                    Cancel
                </button>
            <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
        </div>

    </div>

    
    <div class="space-y-6">

        
        <div class="w-full md:w-1/2">
            <div class="relative">
                <?php if (isset($component)) { $__componentOriginalc3d062a579167d374258253d48d4177f = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc3d062a579167d374258253d48d4177f = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::icon.magnifying-glass','data' => ['class' => 'absolute left-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400 dark:text-gray-500']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::icon.magnifying-glass'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['class' => 'absolute left-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400 dark:text-gray-500']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

<?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc3d062a579167d374258253d48d4177f)): ?>
<?php $attributes = $__attributesOriginalc3d062a579167d374258253d48d4177f; ?>
<?php unset($__attributesOriginalc3d062a579167d374258253d48d4177f); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc3d062a579167d374258253d48d4177f)): ?>
<?php $component = $__componentOriginalc3d062a579167d374258253d48d4177f; ?>
<?php unset($__componentOriginalc3d062a579167d374258253d48d4177f); ?>
<?php endif; ?>
                <input type="text" wire:model.live="search" placeholder="Search by name or email"
                    class="w-full pl-12 pr-4 py-3 text-sm rounded-xl bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition" />
            </div>
        </div>

        
        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::openLoop(); ?><?php endif; ?><?php $__currentLoopData = $this->admins; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $admin): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::startLoop($loop->index); ?><?php endif; ?>
            <div
                class="flex flex-col md:flex-row md:justify-between md:items-center p-6 bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-2xl shadow-sm hover:shadow-md transition duration-300">

                
                <div class="flex flex-col md:flex-row md:items-center md:gap-6">
                    <div class="shrink-0">
                        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if BLOCK]><![endif]--><?php endif; ?><?php if($admin->avatar): ?>
                            <img src="<?php echo e(Storage::disk('public')->url($admin->avatar)); ?>"
                                class="w-12 h-12 rounded-full object-cover shadow" />
                        <?php else: ?>
                            <div
                                class="w-12 h-12 rounded-full bg-gray-200 dark:bg-gray-700 flex items-center justify-center text-gray-500 dark:text-gray-400 font-semibold text-lg">
                                <?php echo e(strtoupper(substr($admin->name, 0, 1))); ?>

                            </div>
                        <?php endif; ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php endif; ?>
                    </div>

                    <div class="mt-3 md:mt-0 flex flex-col gap-1">
                        <span class="font-semibold text-gray-900 dark:text-gray-100 text-lg">
                            <?php echo e($admin->name); ?>

                        </span>
                        <span
                            class="text-sm px-2 py-1 rounded-full w-fit
    <?php echo e($admin->role === 'super_admin' ? 'bg-red-500 text-white' : ($admin->role === 'manager' ? 'bg-green-500 text-white' : 'bg-blue-500 text-white')); ?>">
                            <?php echo e(ucfirst($admin->role)); ?>

                        </span>
                        <span class="text-gray-500 dark:text-gray-400 text-sm">
                            <?php echo e($admin->email); ?>

                        </span>
                    </div>
                </div>

                
                <div class="flex items-center gap-3 mt-4 md:mt-0">

                    
                    <?php if (isset($component)) { $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::button.index','data' => ['wire:click' => 'editAdmin('.e($admin->id).')','variant' => 'primary','color' => 'green','type' => 'button']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['wire:click' => 'editAdmin('.e($admin->id).')','variant' => 'primary','color' => 'green','type' => 'button']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        Edit
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $attributes = $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $component = $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>

                    
                    <?php if (isset($component)) { $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580 = $component; } ?>
<?php if (isset($attributes)) { $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580 = $attributes; } ?>
<?php $component = Illuminate\View\AnonymousComponent::resolve(['view' => 'e60dd9d2c3a62d619c9acb38f20d5aa5::button.index','data' => ['xData' => true,'variant' => 'danger','type' => 'button','@click.prevent' => 'if (confirm(\'Are you sure you want to delete this admin?\')) { $wire.delete('.e($admin->id).') }']] + (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag ? $attributes->all() : [])); ?>
<?php $component->withName('flux::button'); ?>
<?php if ($component->shouldRender()): ?>
<?php $__env->startComponent($component->resolveView(), $component->data()); ?>
<?php if (isset($attributes) && $attributes instanceof Illuminate\View\ComponentAttributeBag): ?>
<?php $attributes = $attributes->except(\Illuminate\View\AnonymousComponent::ignoredParameterNames()); ?>
<?php endif; ?>
<?php $component->withAttributes(['x-data' => true,'variant' => 'danger','type' => 'button','@click.prevent' => 'if (confirm(\'Are you sure you want to delete this admin?\')) { $wire.delete('.e($admin->id).') }']); ?>
<?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::processComponentKey($component); ?>

                        Delete
                     <?php echo $__env->renderComponent(); ?>
<?php endif; ?>
<?php if (isset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $attributes = $__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__attributesOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>
<?php if (isset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580)): ?>
<?php $component = $__componentOriginalc04b147acd0e65cc1a77f86fb0e81580; ?>
<?php unset($__componentOriginalc04b147acd0e65cc1a77f86fb0e81580); ?>
<?php endif; ?>

                </div>
            </div>
        <?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::endLoop(); ?><?php endif; ?><?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><?php if(\Livewire\Mechanisms\ExtendBlade\ExtendBlade::isRenderingLivewireComponent()): ?><!--[if ENDBLOCK]><![endif]--><?php \Livewire\Features\SupportCompiledWireKeys\SupportCompiledWireKeys::closeLoop(); ?><?php endif; ?>

    </div>

</div>
<?php /**PATH /Users/macuser/Herd/Novabank/resources/views/pages/admin/⚡setting.blade.php ENDPATH**/ ?>