<?php

use Livewire\Component;
use App\Models\Account;
use App\Models\Beneficiary;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Collection;

new class extends Component {
    public ?int $selectedBeneficiary = null;
    public Collection $beneficiaries;
    public $allInternationalMethods;
    public ?int $selectedPaymentMethod = null;
    public string $transferType = Beneficiary::LOCAL; // default
    public ?string $selectedPaymentMethodType = null;

    public array $newBeneficiary = [
        'account_number' => '',
        'nickname' => null,
        'recipient_name' => null,
        'beneficiary_name' => null,
        'bank_name' => null,
        'bank_address' => null,
        'country' => null,
        'swift_code' => null,
        'iban' => null,
        'routing_number' => null,
        'currency' => null,
        'coin' => null,
        'network' => null,
        'wallet_address' => null,
        'paypal_email' => null,
        'transaction_hash' => null,
        'reference' => null,
        'note' => null,
    ];

    // -----------------------------
    // Component Lifecycle
    // -----------------------------
    public function mount(string $transferType = Beneficiary::LOCAL): void
    {
        $this->transferType = $transferType;
        $this->beneficiaries = collect();

        // Get all active international methods
        $this->allInternationalMethods = \App\Models\PaymentMethod::where('type', 'international_transfer')->where('is_active', true)->get();

        // Preselect first method for international transfers
        if ($transferType === Beneficiary::INTERNATIONAL && $this->allInternationalMethods->isNotEmpty()) {
            $this->selectedPaymentMethod = $this->allInternationalMethods->first()->id;
            $this->updatedSelectedPaymentMethod();
        }

        $this->refreshBeneficiaries();
    }

    // -----------------------------
    // Computed Properties
    // -----------------------------
    #[Computed]
    public function getCurrentPaymentFieldsProperty(): array
    {
        $method = \App\Models\PaymentMethod::find($this->selectedPaymentMethod);
        return $method->fields['fields'] ?? [];
    }

    // -----------------------------
    // Event Handlers
    // -----------------------------
    public function updatedSelectedPaymentMethod(): void
    {
        $method = \App\Models\PaymentMethod::find($this->selectedPaymentMethod);

        if (!$method) {
            $this->selectedPaymentMethodType = null;
            return;
        }

        $nameLower = strtolower($method->name);

        // Override type for crypto/paypal based on name
        if (str_contains($nameLower, 'crypto')) {
            $this->selectedPaymentMethodType = 'crypto';
        } elseif (str_contains($nameLower, 'paypal')) {
            $this->selectedPaymentMethodType = 'paypal';
        } else {
            $this->selectedPaymentMethodType = $method->type; // e.g., international_transfer, local
        }

        // Clear irrelevant fields for crypto/paypal
        if (in_array($this->selectedPaymentMethodType, ['crypto', 'paypal'])) {
            foreach (['account_number', 'bank_name', 'bank_address', 'swift_code', 'iban', 'routing_number'] as $field) {
                $this->newBeneficiary[$field] = null;
            }
        }
    }

    #[On('beneficiary-added')]
    public function refreshBeneficiaries($id = null): void
    {
        $user = auth()->user();
        $query = $user->beneficiaries();

        $query = $this->transferType === Beneficiary::INTERNATIONAL ? $query->international() : $query->local();

        $this->beneficiaries = $query->get();

        if ($id) {
            $this->selectedBeneficiary = $id;
        }
    }

    // -----------------------------
    // Beneficiary Selection
    // -----------------------------
    public function selectBeneficiary(int $id): void
    {
        $this->selectedBeneficiary = $id;

        $beneficiary = Beneficiary::find($id);
        if (!$beneficiary) {
            return;
        }

        $methods = \App\Models\PaymentMethod::where('type', 'international_transfer')->where('is_active', true)->get();

        $paymentMethod = $methods->first(fn($m) => $m->id === ($beneficiary->meta['payment_method_id'] ?? null)) ?? $methods->first();

        $this->selectedPaymentMethod = $paymentMethod->id ?? null;
        $this->updatedSelectedPaymentMethod();

        // Autofill newBeneficiary fields
        foreach ($beneficiary->meta ?? [] as $key => $value) {
            $this->newBeneficiary[$key] = $value;
        }

        $this->newBeneficiary['account_number'] = $beneficiary->account_number;
        $this->newBeneficiary['nickname'] = $beneficiary->nickname;

        $this->dispatch('beneficiary-selected', id: $id, paymentMethodName: $paymentMethod->name ?? null);
    }

    // -----------------------------
    // Beneficiary Management
    // -----------------------------
    public function addBeneficiary(): void
    {
        $this->validate($this->getValidationRules());

        $user = auth()->user();
        $acctNum = $this->newBeneficiary['account_number'];

        // Prevent adding user's own account
        $userAccountNumbers = $user->profile?->accounts()->pluck('account_number')->toArray() ?? [];
        if (in_array($acctNum, $userAccountNumbers)) {
            throw ValidationException::withMessages([
                'newBeneficiary.account_number' => 'You cannot add your own account as a beneficiary.',
            ]);
        }

        // Validate local account exists
        if ($this->transferType === Beneficiary::LOCAL) {
            $account = $this->checkAccountNumber($acctNum);
            if (!$account?->profile?->user) {
                throw ValidationException::withMessages([
                    'newBeneficiary.account_number' => 'Account number not found.',
                ]);
            }
        }

        // Build meta
        $meta = collect($this->newBeneficiary)
            ->except(['account_number', 'nickname'])
            ->filter(fn($v) => $v !== null)
            ->toArray();

        $beneficiary = Beneficiary::create([
            'user_id' => $user->id,
            'nickname' => $this->newBeneficiary['nickname'] ?? null,
            'account_number' => $acctNum,
            'type' => $this->transferType,
            'meta' => array_merge(['payment_method_id' => $this->selectedPaymentMethod], $meta),
        ]);

        $this->beneficiaries->push($beneficiary);
        $this->reset(['newBeneficiary']);
        $this->resetErrorBag();

        $this->dispatch('beneficiary-added', id: $beneficiary->id);
        $this->dispatch('showToast', message: 'Beneficiary added successfully.', type: 'success');
    }

    // -----------------------------
    // Validation
    // -----------------------------
    protected function getValidationRules(): array
    {
        $rules = ['newBeneficiary.nickname' => ['nullable', 'string', 'max:255']];
        $excludedTypes = ['crypto', 'paypal'];

        if ($this->transferType === Beneficiary::LOCAL || !in_array($this->selectedPaymentMethodType, $excludedTypes)) {
            $rules['newBeneficiary.account_number'] = ['required', 'numeric', Rule::unique('beneficiaries', 'account_number')->where('user_id', auth()->id())];
        }

        foreach ($this->currentPaymentFields as $field) {
            $key = 'newBeneficiary.' . $field['name'];
            if (empty($field['required'])) {
                $rules[$key] = 'nullable';
                continue;
            }

            $rules[$key] = match ($field['type'] ?? 'text') {
                'text', 'textarea' => 'required|string|max:255',
                'number' => 'required|numeric|min:1',
                'select' => ['required', Rule::in(array_map(fn($o) => $o['value'] ?? $o, $field['options'] ?? []))],
                default => 'required|string|max:255',
            };
        }

        return $rules;
    }

    protected function getValidationMessages(): array
    {
        $messages = [];
        foreach ($this->currentPaymentFields as $field) {
            $key = 'newBeneficiary.' . $field['name'];
            $label = $field['label'] ?? ucfirst(str_replace('_', ' ', $field['name']));

            if ($field['type'] === 'file') {
                $messages["$key.required"] = "$label is required.";
                $messages["$key.file"] = "$label must be a valid file.";
                $messages["$key.mimes"] = "$label must be jpg, png or pdf.";
                $messages["$key.max"] = "$label may not exceed 2MB.";
            } elseif ($field['type'] === 'number') {
                $messages["$key.required"] = "$label is required.";
                $messages["$key.numeric"] = "$label must be numeric.";
                $messages["$key.min"] = "$label must be at least 1.";
            } else {
                $messages["$key.required"] = "$label is required.";
                $messages["$key.string"] = "$label must be a valid string.";
                $messages["$key.max"] = "$label may not exceed 255 characters.";
            }
        }

        // Static messages
        $messages['newBeneficiary.account_number.required'] = 'Account number is required.';
        $messages['newBeneficiary.account_number.numeric'] = 'Account number must be numeric.';
        $messages['newBeneficiary.account_number.unique'] = 'You already added this account as a beneficiary.';

        return $messages;
    }

    private function checkAccountNumber(string $accountNumber): ?Account
    {
        return Account::where('account_number', $accountNumber)->with('profile.user')->first();
    }

    public function closeModal(): void
    {
        $this->reset(['newBeneficiary']);
        $this->resetErrorBag();
    }
};
?>
<div class="rounded-xl p-4 shadow-sm dark:shadow-gray-900/25 border border-gray-100 dark:border-gray-700">

    <!-- Heading -->
    <div class="flex items-center justify-between text-gray-800 dark:text-gray-200 mb-4">
        <h3 class="text-base font-semibold text-gray-900 dark:text-white">{{ $title ?? 'Beneficiaries' }}</h3>
    </div>

    <!-- Beneficiaries Flex Section -->
    <div x-data="{ showAll: false }" class="flex space-x-3 overflow-x-auto pb-2">

        <!-- Add Beneficiary Button -->
        <button type="button" x-data @click="$dispatch('open-modal')"
            class="shrink-0 flex flex-col items-center justify-center group">
            <div
                class="w-16 h-16 rounded-full border-2 border-dashed border-gray-300 dark:border-gray-600 bg-gray-50 dark:bg-gray-800 flex items-center justify-center mb-2 group-hover:bg-gray-100 dark:group-hover:bg-gray-700 group-hover:border-primary-400 dark:group-hover:border-primary-500 transition-all duration-200">
                <x-flux::icon.plus variant="solid" class="w-5 h-5" />
            </div>
            <span
                class="text-xs text-gray-500 dark:text-gray-400 text-center group-hover:text-primary-600 dark:group-hover:text-primary-400 transition-colors">
                Add New
            </span>
        </button>

        <!-- Existing Beneficiaries -->
        @forelse ($beneficiaries as $index => $beneficiary)
            <div x-show="showAll || {{ $index }} < 6" x-transition:enter="transition ease-out duration-300"
                x-transition:enter-start="opacity-0 translate-y-2" x-transition:enter-end="opacity-100 translate-y-0"
                x-transition:leave="transition ease-in duration-200"
                x-transition:leave-start="opacity-100 translate-y-0" x-transition:leave-end="opacity-0 translate-y-2"
                class="shrink-0 flex flex-col items-center justify-center group relative">

                <button wire:click="selectBeneficiary({{ $beneficiary->id }})"
                    class="relative w-16 h-16 rounded-full flex items-center justify-center mb-2 border-2 border-white dark:border-gray-800 shadow-sm dark:shadow-gray-900/25 hover:scale-105 transition-all duration-200 {{ $beneficiary->color }} {{ $selectedBeneficiary === $beneficiary->id ? 'ring-4 ring-sky-500' : '' }}"
                    title="{{ ($beneficiary->meta['recipient_name'] ?? $beneficiary->nickname) . ' - Acct: ' . $beneficiary->account_number }}">
                    {{ $beneficiary->initials }}
                    @if ($selectedBeneficiary === $beneficiary->id)
                        <span
                            class="absolute bottom-0 -right-1 w-5 h-5 bg-green-500 rounded-full flex items-center justify-center text-white text-xs shadow-lg">
                            ✓
                        </span>
                    @endif
                </button>
            </div>
        @empty
            <div class="shrink-0 flex flex-col items-center justify-center ">
                <div class="w-16 h-16 rounded-full bg-gray-100 dark:bg-gray-700 flex items-center justify-center mb-3">
                    <x-flux::icon.user-group variant="solid" class="w-5 h-5" />
                </div>
                <p class="text-sm text-gray-500 dark:text-gray-400 text-center">No saved beneficiaries</p>
                <p class="text-xs text-gray-400 dark:text-gray-500 text-center mt-1">Add one to get started
                </p>
            </div>
        @endforelse

        <!-- Show More Button -->
        @if ($beneficiaries->count() > 6)
            <button @click="showAll = !showAll"
                class="flex items-center justify-center w-14 h-14 rounded-full border border-gray-300 dark:border-gray-600 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:ring hover:ring-sky-200 dark:hover:ring-sky-700 transition">
                <span x-text="showAll ? '-' : '+'"></span>
            </button>
        @endif

    </div>

    <!-- Add Beneficiary Modal -->
    <div x-data="{ open: false }" x-cloak x-show="open" x-on:open-modal.window="open = true"
        x-on:beneficiary-added.window="open = false" x-on:keydown.escape.window="open = false; $wire.closeModal()"
        class="fixed inset-0 z-50 flex items-center justify-center bg-black/60">
        <div @click.away="open = false; $wire.closeModal()"
            class="bg-white dark:bg-gray-800 rounded-xl shadow-lg w-full max-w-3xl p-6 max-h-[90vh] overflow-auto">

            <h2 class="text-lg font-semibold mb-4 text-gray-900 dark:text-white">
                Add Beneficiary
            </h2>

            <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 pb-4">
                <!-- Local/common fields -->
                @if ($selectedPaymentMethodType !== 'crypto' && $selectedPaymentMethodType !== 'paypal')
                    <x-flux::input wire:model.defer="newBeneficiary.account_number" label="Account Number"
                        type="number" />
                @endif

                <x-flux::input wire:model.defer="newBeneficiary.nickname" label="Nickname" type="text" />

                <!-- International fields -->
                @if ($transferType === Beneficiary::INTERNATIONAL)

                    <flux:select wire:model.live.debounce.500ms="selectedPaymentMethod" label="Transfer Method">
                        <option value="" disabled>Select Method</option>
                        @foreach ($allInternationalMethods as $method)
                            <option value="{{ $method->id }}">{{ $method->name }}</option>
                        @endforeach
                    </flux:select>

                    @foreach ($this->currentPaymentFields as $field)
                        @php
                            $fieldName = $field['name'];
                            $isTextarea = $field['type'] === 'textarea';
                        @endphp

                        @if ($field['type'] === 'text')
                            <x-flux::input wire:model.defer="newBeneficiary.{{ $fieldName }}"
                                label="{{ $field['label'] }}" type="text" />
                        @elseif ($isTextarea)
                            <x-flux::textarea wire:model.defer="newBeneficiary.{{ $fieldName }}"
                                label="{{ $field['label'] }}" class="col-span-2 w-full" />
                        @elseif ($field['type'] === 'select')
                            <flux:select wire:model="newBeneficiary.{{ $fieldName }}"
                                label="{{ $field['label'] }}">
                                <option value="" disabled>Select {{ $field['label'] }}</option>
                                @foreach ($field['options'] ?? [] as $option)
                                    <option value="{{ $option['value'] }}">
                                        {{ $option['label'] }}
                                    </option>
                                @endforeach
                            </flux:select>
                        @endif
                    @endforeach
                @endif

            </div>

            <div class="mt-4 flex justify-end gap-2">
                <button type="button" @click="open = false; $wire.closeModal()"
                    class="px-3 py-1 rounded bg-gray-200 dark:bg-gray-600">
                    Cancel
                </button>

                <x-button type="button" wire:click="addBeneficiary" wire:loading.attr="disabled"
                    wire:target="addBeneficiary" class="px-3 py-1 rounded bg-sky-600 text-white disabled:opacity-50">
                    Save Beneficiary
                </x-button>
            </div>
        </div>
    </div>

</div>
