<?php

use App\Models\User;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public bool $otpEnabled = false;
    public User $user;

    public function mount()
    {
        $this->user = Auth::user();
        $this->otpEnabled = $this->user->otpEnabled(); // uses the model helper
    }

    public function toggleAuth()
    {
        if (!$this->otpEnabled && $this->user->hasActiveTrustedDevices()) {
            $this->dispatch('showToast', message: 'You must revoke all trusted devices before enabling two-factor authentication.', type: 'warning');

            return;
        }
        $this->otpEnabled = !$this->otpEnabled;

        // ✅ Update the correct database column
        $this->user->update([
            'otp_login_enabled' => $this->otpEnabled,
        ]);

        // Dispatch toast
        $message = $this->otpEnabled ? 'Two-Factor Authentication enabled!' : 'Two-Factor Authentication disabled!';

        $this->dispatch('showToast', message: $message, type: 'success');
    }
};
?>


<x-slot:title>
    {{ config('app.name') }} || Authentication Management
</x-slot:title>

<section class="w-full py-5">
    <x-settings.layout :heading="__('Two-Factor Authentication')" :subheading="__('Add an extra layer of security to your account')">

        <div class="grid grid-cols-1 gap-4">
            <!-- Main Content Area -->
            <div
                class="bg-white/80 dark:bg-gray-800/80 backdrop-blur-xl rounded-2xl shadow-lg dark:shadow-gray-900/50 border border-white/20 dark:border-gray-700/50 overflow-hidden">
                <!-- Content Header -->
                <div class="border-b border-gray-200/50 dark:border-gray-700/50 px-4 py-3 flex items-center">
                    <div class="w-6 h-6 bg-sky-500 rounded-lg flex items-center justify-center mr-2">
                        <x-flux::icon.shield-exclamation variant="solid" class="text-white w-4 h-4" />
                    </div>
                    <div>
                        <h2 class="text-sm font-bold text-gray-900 dark:text-white">
                            Email-Based Two-Factor Authentication
                        </h2>
                        <p class="text-xs text-gray-500 dark:text-gray-400">
                            Add an extra layer of security to your account
                        </p>
                    </div>
                </div>

                <div class="p-4">
                    <!-- Status Badge -->
                    <div class="flex items-center mb-3">
                        <span
                            class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium
                                        {{ $otpEnabled
                                            ? 'bg-green-100 dark:bg-green-700/50 text-green-700 dark:text-green-300'
                                            : 'bg-gray-100 dark:bg-gray-700/50 text-gray-700 dark:text-gray-300' }}">
                            <x-flux::icon.shield-exclamation variant="solid" class="w-4 h-4 mr-1" />
                            {{ $otpEnabled ? 'Enabled' : 'Disabled' }}
                        </span>
                        <span class="ml-2 text-gray-500 dark:text-gray-400 text-xs">Current Status</span>
                    </div>

                    <p class="text-gray-600 dark:text-gray-300 mb-3 text-xs">
                        When enabled, a 6-digit verification code will be sent to your email address
                        <strong class="text-sky-600 dark:text-sky-400">{{ $authUser->email }}</strong> each time you
                        log in.
                    </p>

                    <!-- How it Works Card -->
                    <div
                        class="bg-blue-50 dark:bg-blue-900/20 rounded-xl p-3 border border-blue-100 dark:border-blue-800/50 mb-4">
                        <div class="flex">
                            <div class="shrink-0">
                                <x-flux::icon.information-circle variant="solid" class="text-white w-4 h-4" />
                            </div>
                            <div class="ml-2 text-xs">
                                <h3 class="font-medium text-blue-800 dark:text-blue-300">
                                    How Two-Factor Authentication Works
                                </h3>
                                <ul class="list-disc pl-3 mt-1 space-y-0.5 text-blue-700 dark:text-blue-400">
                                    <li>When you log in with your password, a 6-digit code will be sent to your
                                        email</li>
                                    <li>You must enter this code to complete your login</li>
                                    <li>This adds an extra layer of security to your account</li>
                                    <li>The code expires after 10 minutes for security</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    @if (!$otpEnabled && $user->hasActiveTrustedDevices())
                        <div
                            class="mb-4 rounded-lg border border-yellow-200 bg-yellow-50 dark:bg-yellow-900/20 p-3 text-xs text-yellow-800 dark:text-yellow-300">
                            <strong>Action required:</strong>
                            You must revoke all trusted devices before enabling two-factor authentication.
                            <a href="{{ route('user.settings.trusted.device') }}"
                                class="inline-flex items-center ml-1 underline font-medium text-sky-700 dark:text-sky-400">
                                <x-flux::icon.device-phone-mobile class="w-3 h-3 mr-1" />
                                Manage trusted devices
                            </a>

                        </div>
                    @endif


                    <!-- Toggle Button -->
                    <x-button wire:click="toggleAuth" type="button" icon="shield-check" :disabled="!$otpEnabled && $user->hasActiveTrustedDevices()">
                        {{ $otpEnabled ? __('Disable Two-Factor Authentication') : __('Enable Two-Factor Authentication') }}
                    </x-button>

                </div>
            </div>

            <!-- Security Tips Sidebar -->
            <div
                class="bg-white/80 dark:bg-gray-800/80 backdrop-blur-xl rounded-2xl shadow-lg dark:shadow-gray-900/50 border border-white/20 dark:border-gray-700/50 overflow-hidden">
                <!-- Header -->
                <div class="border-b border-gray-200/50 dark:border-gray-700/50 px-4 py-3 flex items-center">
                    <div class="w-6 h-6 bg-sky-500 rounded-lg flex items-center justify-center mr-2">
                        <x-flux::icon.light-bulb variant="solid" class="text-white w-4 h-4" />
                    </div>
                    <h3 class="text-sm font-bold text-gray-900 dark:text-white">Security Tips</h3>
                </div>

                <!-- Tips List -->
                <div class="p-4 text-xs space-y-2 text-gray-700 dark:text-gray-300">
                    <ul class="space-y-2">
                        <li class="flex items-start">
                            <x-flux::icon.check-badge variant="solid" class="w-4 h-4 mr-1" />
                            Use a strong, unique password
                        </li>
                        <li class="flex items-start">
                            <x-flux::icon.check-badge variant="solid" class="w-4 h-4 mr-1" />
                            Enable two-factor authentication
                        </li>
                        <li class="flex items-start">
                            <x-flux::icon.check-badge variant="solid" class="w-4 h-4 mr-1" />
                            Keep your email secure
                        </li>
                        <li class="flex items-start">
                            <x-flux::icon.check-badge variant="solid" class="w-4 h-4 mr-1" />
                            Log out when using shared devices
                        </li>
                        <li class="flex items-start">
                            <x-flux::icon.check-badge variant="solid" class="w-4 h-4 mr-1" />
                            Regularly check your account activity
                        </li>
                    </ul>
                </div>
            </div>
        </div>
    </x-settings.layout>
</section>
