<?php

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

new class extends Component {
    public $current_password = '';
    public $new_password = '';
    public $new_password_confirmation = '';

    public function changePassword()
    {
        $this->validate(
            [
                'current_password' => ['required', 'string', 'current_password'],
                'new_password' => ['required', 'confirmed', Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised()],
            ],
            [
                'current_password.required' => 'Please enter your current password.',
                'current_password.current_password' => 'The current password is incorrect.',
                'new_password.required' => 'Please enter a new password.',
                'new_password.confirmed' => 'New password and confirmation do not match.',
                'new_password.min' => 'Your new password must be at least :min characters long.',
                'new_password.letters' => 'Your new password must contain at least one letter.',
                'new_password.mixedCase' => 'Your new password must include both uppercase and lowercase letters.',
                'new_password.numbers' => 'Your new password must contain at least one number.',
                'new_password.symbols' => 'Your new password must include at least one symbol.',
                'new_password.uncompromised' => 'This password has appeared in a data leak. Please choose a different one.',
            ],
        );

        $user = Auth::user();
        // Check current password manually
        if (!Hash::check($this->current_password, $user->password)) {
            $this->addError('current_password', 'Current password is incorrect.');
            return;
        }

        $user->password = Hash::make($this->new_password);
        $user->save();

        // Log the user out after password change (optional)
        Auth::logout();
        session()->invalidate();
        session()->regenerateToken();

        return redirect()
            ->route('login')
            ->with([
                'message' => 'Your password has been changed successfully. Please log in again!.',
                'type' => 'success',
            ]);
    }
};
?>
<x-slot:title>
    {{ config('app.name') }} || Update Profile Password
</x-slot:title>

<section class="w-full py-5">
    <x-settings.layout :heading="__('Change Password
                                                                                                                                                                                                                                                                        ')" :subheading="__('Update your account password to maintain security')">
        <form wire:transition wire:submit='changePassword' class="space-y-4">
            <x-flux::input size="sm" wire:model.defer='current_password' placeholder="enter your current password"
                icon="lock-closed" label="Current Password" viewable type="password" />
            <flux:input type="password" size="sm" icon="key" label="Password" wire:model.defer="new_password"
                :placeholder="__('Password')" viewable />
            <flux:input size="sm" icon="key" wire:model.defer="new_password_confirmation"
                :label="__('Confirm Password')" type="password" autocomplete="confirm-password"
                :placeholder="__('Confirm Password')" viewable />

            <div class="bg-blue-50 dark:bg-blue-900/20 rounded-xl p-3 border border-blue-100 dark:border-blue-800/50">
                <h4 class="text-blue-800 dark:text-blue-300 font-medium mb-2 flex items-center text-xs">
                    <i class="fa-solid fa-shield text-blue-600 dark:text-blue-400 text-xs mr-1"></i>
                    Password Requirements
                </h4>
                <p class="text-xs text-blue-600 dark:text-blue-400 mb-2">Ensure that these requirements are met:</p>
                <ul class="space-y-1">
                    <li class="flex items-center text-xs text-blue-700 dark:text-blue-300">
                        <div class="h-4 w-4 mr-2 flex items-center justify-center">
                            <div class="h-1.5 w-1.5 bg-blue-500 rounded-full"></div>
                        </div>
                        Minimum 8 characters long - the more, the better
                    </li>
                    <li class="flex items-center text-xs text-blue-700 dark:text-blue-300">
                        <div class="h-4 w-4 mr-2 flex items-center justify-center">
                            <div class="h-1.5 w-1.5 bg-blue-500 rounded-full"></div>
                        </div>
                        At least one lowercase character
                    </li>
                    <li class="flex items-center text-xs text-blue-700 dark:text-blue-300">
                        <div class="h-4 w-4 mr-2 flex items-center justify-center">
                            <div class="h-1.5 w-1.5 bg-blue-500 rounded-full"></div>
                        </div>
                        At least one uppercase character
                    </li>
                    <li class="flex items-center text-xs text-blue-700 dark:text-blue-300">
                        <div class="h-4 w-4 mr-2 flex items-center justify-center">
                            <div class="h-1.5 w-1.5 bg-blue-500 rounded-full"></div>
                        </div>
                        At least one number
                    </li>
                </ul>
            </div>

            <div
                class="bg-yellow-50 dark:bg-yellow-900/20 rounded-xl p-3 border border-yellow-100 dark:border-yellow-800/50">
                <div class="flex">
                    <div class="shrink-0">
                        <i class="fa-solid fa-exclamation-triangle text-yellow-500 text-sm"></i>
                    </div>
                    <div class="ml-2">
                        <h3 class="text-xs font-medium text-yellow-800 dark:text-yellow-300">Security Reminder</h3>
                        <p class="mt-1 text-xs text-yellow-700 dark:text-yellow-400">
                            After changing your password, you'll be required to log in again with your new credentials.
                        </p>
                    </div>
                </div>
            </div>

            <x-button type="submit" icon="cloud-arrow-down" class="w-full">update password</x-button>
        </form>
    </x-settings.layout>


</section>
