<?php

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

new class extends Component {
    /**
     * Delete the currently authenticated user.
     */
    public string $password = '';

    public function confirmDelete()
    {
        // Validate password first
        $this->validate(['password' => 'required']);

        $user = Auth::user();

        if (!Hash::check($this->password, $user->password)) {
            $this->addError('password', 'Password is incorrect.');
            return;
        }

        $this->dispatch('open-delete-prompt');
    }

    public function delete()
    {
        $user = Auth::user();

        // Delete the user
        $user->delete();

        Auth::logout();
        session()->invalidate();
        session()->regenerateToken();

        return redirect()
            ->route('login')
            ->with([
                'message' => 'Your account has been deleted successfully!',
                'type' => 'success',
            ]);
    }
};
?>

<section class="mt-10 space-y-6">
    <div class="relative mb-5">
        <flux:heading>{{ __('Delete account') }}</flux:heading>
        <flux:subheading>{{ __('Delete your account and all of its resources') }}</flux:subheading>
    </div>



    <x-button icon="trash" type="button" @click="$dispatch('open-delete-modal')"
        class="flex items-center gap-2 bg-linear-to-bl from-red-500 to-red-800 p-3 rounded-md">
        <span class="text-sm capitalize font-semibold text-white"> {{ __('Delete account') }}</span>
    </x-button>

    <x-modal @open-delete-modal.window="openModal()" @close-delete-modal.window="close()"
        x-on:delete-success.window="close()">
        <div class="space-y-6">
            <!-- Header -->
            <div class="text-center space-y-4">
                <div
                    class="w-14 h-14 mx-auto flex items-center justify-center rounded-xl bg-linear-to-br from-red-500 to-red-700 text-white">
                    <x-flux::icon.trash variant="solid" class="w-6 h-6" />
                </div>

                <h2 class="text-lg font-bold text-gray-900 dark:text-white">
                    {{ __('Are you sure you want to delete your account?') }}
                </h2>

            </div>

            <!-- Password Input -->
            <x-input wire:model.defer="password" type="password" viewable placeholder="Enter your password"
                class="w-full" />

            <!-- Action Buttons -->
            <div class="space-y-3">
                <x-button wire:click="confirmDelete" type="button"
                    class="w-full bg-sky-600 hover:bg-sky-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-sky-500 text-white font-semibold transition-colors">
                    {{ __('Delete account') }}
                </x-button>

                <button type="button" @click="$dispatch('close-delete-modal')"
                    class="w-full text-center text-sm text-red-600 hover:underline font-medium">
                    {{ __(key: 'Cancel') }}
                </button>
            </div>
    </x-modal>

    <x-modal @open-delete-prompt.window="openModal()" @close-delete-prompt.window="close()"
        x-on:delete-success.window="close()">
        <div class="space-y-6">
            <!-- Header -->
            <div class="text-center space-y-4">
                <div
                    class="w-14 h-14 mx-auto flex items-center justify-center rounded-xl bg-linear-to-br from-red-500 to-red-700 text-white">
                    <x-flux::icon.trash variant="solid" class="w-6 h-6" />
                </div>

                <h2 class="text-lg font-bold text-gray-900 dark:text-white">
                    {{ __('Are you absolutely sure?') }}
                </h2>
                <p class="text-sm text-gray-600 dark:text-gray-300">
                    {{ __('Once your account is deleted, all of its resources and data will be permanently removed. You will be logged out immediately.') }}
                </p>

            </div>

            <!-- Action Buttons -->
            <div class="flex space-x-2">
                <x-button wire:click="delete" variant="danger" class="bg-red-600 hover:bg-red-700 text-white w-full">
                    {{ __('Delete account') }}
                </x-button>
                <button type="button" @click="$dispatch('close-delete-prompt')"
                    class="w-full text-center text-sm text-red-600 hover:underline font-medium">
                    {{ __(key: 'Cancel') }}
                </button>
            </div>
    </x-modal>

</section>
