<?php

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use App\Traits\WithTableHelpers;
use App\Models\Loan;
use App\Support\Currency;
use App\Services\ExportService;

new class extends Component {
    use WithPagination, WithTableHelpers;

    // Pagination
    public int $perPage = 10;
    public int $page = 1;

    // Export settings
    public string $exportFormat = 'csv'; // csv | xlsx | pdf
    public string $exportDelivery = 'download'; // download | email
    public string $exportScope = 'page'; // page | all

    public string $currencySymbol = '$';
    public ?string $modalMessage = null;

    protected function exportService(): ExportService
    {
        return app(ExportService::class);
    }

    public function clearModalMessage()
    {
        $this->modalMessage = null;
    }

    public function export()
    {
        // Determine which loans to export based on scope
        $loans =
            $this->exportScope === 'page'
                ? $this->loans->getCollection() // current page only
                : $this->baseQuery()->get(); // all loans

        // Prevent full PDF export if needed
        if ($this->exportFormat === 'pdf' && $this->exportScope === 'all') {
            $this->exportScope = 'page';
        }

        // Show warning if no data
        if ($loans->isEmpty()) {
            $this->dispatch('showToast', message: 'No loans to export.', type: 'warning');
            return;
        }

        $exportTitle = 'Loans Export';

        // Export via download
        if ($this->exportDelivery === 'download') {
            return $this->exportService()->download(
                $loans,
                $this->exportFormat,
                'loans', // filename
                pdfTitle: $exportTitle,
            );
        }

        // Export via email
        $this->exportService()->email($loans, $this->exportFormat, 'loans', auth()->user()->email, pdfTitle: $exportTitle);

        $this->dispatch('showToast', message: 'Export emailed successfully.', type: 'success');
    }

    /**
     * Searchable fields
     */
    protected function searchableFields(): array
    {
        return [
            'id', // Loan ID
            'amount', // Loan amount
            'total_interest', // Interest
            'monthly_repayment', // Monthly repayment
            'total_repayment', // Total repayment
            'principal_outstanding', // Remaining principal
            'interest_outstanding', // Remaining interest
            'penalty_outstanding', // Remaining penalty
            'duration', // Duration in months
            'status', // Loan status
            'created_at', // Created date
            'first_due_date', // First due date
            'purpose', // Loan purpose
        ];
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->dispatch('perPageUpdated', value: $this->perPage);
    }

    public function mount(ExportService $exportService)
    {
        $this->exportService = $exportService;

        $user = auth()->user();
        $this->currencySymbol = $user->profile ? Currency::symbol($user->profile->currency) : '$';
    }

    /**
     * Base query
     */
    protected function baseQuery()
    {
        $profile = auth()->user()->profile;

        if (!$profile) {
            return Loan::query()->whereRaw('1=0');
        }

        return Loan::query()
            ->whereIn('account_id', $profile->accounts()->select('id'))
            ->with('account')
            ->tap(fn($q) => $this->applySearch($q, $this->searchableFields()))
            ->orderBy($this->sortField, $this->sortDirection);
    }

    /**
     * Paginated loans
     */
    #[Computed]
    public function loans()
    {
        return $this->baseQuery()->paginate($this->perPage);
    }
};
?>

<div class="space-y-5 py-5">

    <x-table.wrapper :items="$this->loans">
        {{-- SEARCH --}}
        <x-slot name="search">
            <div class="flex items-center justify-between gap-4">

                <div class="w-full md:w-1/3">
                    <div class="relative">
                        <x-flux::icon.magnifying-glass
                            class="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400 dark:text-gray-500" />
                        <input type="text" wire:model.live="search" placeholder="Search by model or title"
                            class="w-full pl-10 pr-4 py-2 text-sm rounded-md bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-sky-600 focus:border-sky-600 transition" />
                    </div>
                </div>
                <div class="relative block">
                    <button type="button" @click="$dispatch('open-export-modal')"
                        class="flex items-center gap-2 bg-linear-to-bl from-sky-500 to-sky-800 p-3 rounded-2xl">
                        <x-flux::icon.arrow-down-tray class="w-5 h-5 text-white" variant="solid" />
                        <span class="text-sm capitalize font-semibold text-white">export</span>
                    </button>

                    <!-- EXPORT MODAL -->
                    <x-modal @open-export-modal.window="openModal()" @close-export-modal.window="close()"
                        x-on:export-finished.window="close()">
                        <div class="space-y-6">
                            <!-- Header -->
                            <div class="text-center space-y-2">
                                <div
                                    class="w-12 h-12 bg-linear-to-bl from-sky-500 to-sky-800 rounded-xl mx-auto flex items-center justify-center text-white">
                                    <x-flux::icon.arrow-down-tray variant="solid" class="w-5 h-5" />
                                </div>
                                <h2 class="text-base font-semibold tracking-wide capitalize">Export Transactions
                                </h2>
                                <span class="text-sm text-gray-600 dark:text-sky-200">
                                    Download or receive your transaction data
                                </span>
                            </div>

                            <!-- Format -->
                            <x-select wire:model.defer="exportFormat" label="Format" placeholder="===Select Format===">
                                <option value="csv">CSV</option>
                                <option value="xlsx">Excel</option>
                                <option value="pdf">PDF</option>
                            </x-select>

                            <!-- Delivery -->
                            <x-select wire:model.defer="exportDelivery" label="Send As"
                                placeholder="===Select Export Type===">
                                <option value="download">Download</option>
                                <option value="email">Email</option>
                            </x-select>

                            <!-- Feedback -->
                            <div x-show="$wire.modalMessage" x-text="$wire.modalMessage" x-init="$watch('$wire.modalMessage', v => v && setTimeout(() => $wire.clearModalMessage(), 3000))"
                                class="bg-sky-50 dark:bg-gray-700 p-3 rounded-md text-center text-sm text-sky-700 dark:text-sky-200">
                            </div>

                            <x-button wire:click="export" variant="primary" class="w-full bg-sky-600 hover:bg-sky-900"
                                type="button">
                                Export Transaction
                            </x-button>
                            <button type="button" @click="$dispatch('close-export-modal')"
                                class="w-full text-center text-sm text-red-600 hover:underline">
                                Cancel
                            </button>

                        </div>
                    </x-modal>
                </div>

            </div>

        </x-slot>

        {{-- HEADERS --}}
        <x-slot name="head">
            <x-table.head label="Loan ID" field="id" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Amount" field="amount" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Interest" field="total_interest" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Monthly Repayment" />
            <x-table.head label="Total Repayment" />
            <x-table.head label="Outstanding Balance" />
            <x-table.head label="Duration (months)" />
            <x-table.head label="Status" field="status" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Created" field="created_at" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="First Due" field="first_due_date" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Action" />
        </x-slot>

        {{-- BODY (DESKTOP) --}}
        <x-slot name="body">
            @forelse ($this->loans as $loan)
                @php
                    $user = $loan->user;
                    $account = $loan->account;
                    $profile = $account?->profile;

                    $statusClasses = match ($loan->status) {
                        'approved' => 'bg-green-500 text-white',
                        'pending' => 'bg-yellow-400 text-black',
                        'rejected' => 'bg-red-500 text-white',
                        'overdue' => 'bg-red-600 text-white',
                        'completed' => 'bg-blue-500 text-white',
                        'defaulted' => 'bg-gray-500 text-white',
                        default => 'bg-gray-300 text-black',
                    };
                @endphp

                <x-table.row wire:key="row-{{ $loan->id }}"
                    class="hover:shadow-lg transition-shadow duration-200 rounded-lg mb-2">

                    {{-- Loan ID --}}
                    <x-table.cell>{{ $loan->id ?? '_' }}</x-table.cell>

                    {{-- Amount --}}
                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->amount, 2) }}</x-table.cell>

                    {{-- Interest --}}
                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->total_interest, 2) }}</x-table.cell>

                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->monthly_repayment, 2) }}</x-table.cell>
                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->total_repayment, 2) }}</x-table.cell>
                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->remaining_balance, 2) }}</x-table.cell>
                    <x-table.cell>{{ $loan->duration }}</x-table.cell>
                    {{-- Status --}}
                    <x-table.cell>
                        <span class="px-2 py-1 rounded-full text-xs {{ $statusClasses }}">
                            {{ ucfirst($loan->status) }}
                        </span>
                    </x-table.cell>
                    {{-- Created date --}}
                    <x-table.cell>{{ $loan->created_at?->format('d M Y') ?? '-' }}</x-table.cell>

                    {{-- First Due --}}
                    <x-table.cell>{{ $loan->first_due_date?->format('d M Y') ?? '-' }}</x-table.cell>

                    {{-- Actions --}}
                    <x-table.cell>
                        <a href="{{ route('user.loans.show', $loan) }}"
                            class="inline-flex items-center px-3 py-1.5 text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 transition">
                            View
                        </a>

                    </x-table.cell>

                </x-table.row>
            @empty
                <tr>
                    <td colspan="8" class="px-6 py-10 text-center text-gray-400 dark:text-gray-500">
                        No loans found
                    </td>
                </tr>
            @endforelse
        </x-slot>

        {{-- MOBILE --}}
        <x-slot name="mobile">
            @forelse ($this->loans as $loan)
                @php
                    $user = $loan->user;
                    $account = $loan->account;
                    $profile = $account?->profile;

                    $statusClasses = match ($loan->status) {
                        'approved' => 'bg-green-100 text-green-800',
                        'pending' => 'bg-yellow-100 text-yellow-800',
                        'rejected' => 'bg-red-100 text-red-800',
                        'overdue' => 'bg-red-200 text-red-900',
                        'completed' => 'bg-blue-100 text-blue-800',
                        'defaulted' => 'bg-gray-200 text-gray-800',
                        default => 'bg-gray-100 text-gray-700',
                    };
                @endphp

                <x-table.mobile-card class="rounded-xl shadow-md p-4 mb-4 hover:shadow-xl transition duration-200">

                    <div class="flex flex-col text-sm text-gray-700 dark:text-gray-300 space-y-1">
                        <div><strong>Loan ID:</strong> {{ $loan->id ?? '_' }}</div>
                        <div><strong>Amount:</strong> {{ $currencySymbol }}{{ number_format($loan->amount, 2) }}</div>
                        <div><strong>Interest:</strong>
                            {{ $currencySymbol }}{{ number_format($loan->total_interest, 2) }}</div>
                        <div><strong>Created:</strong> {{ $loan->created_at?->format('d M Y') ?? '-' }}</div>
                        <div><strong>First Due:</strong> {{ $loan->first_due_date?->format('d M Y') ?? '-' }}</div>
                        <div><strong>Status:</strong>
                            <span
                                class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $statusClasses }}">
                                {{ ucfirst($loan->status) }}
                            </span>
                        </div>
                        <div><strong>Monthly:</strong>
                            {{ $currencySymbol }}{{ number_format($loan->monthly_repayment, 2) }}</div>
                        <div><strong>Total:</strong>
                            {{ $currencySymbol }}{{ number_format($loan->total_repayment, 2) }}</div>
                        <div><strong>Remaining:</strong>
                            {{ $currencySymbol }}{{ number_format($loan->remaining_balance, 2) }}</div>
                        <div><strong>Next Due:</strong>
                            {{ $loan->installments()->where('status', 'pending')->orderBy('due_date')->first()?->due_date?->format('d M Y') ?? '-' }}
                        </div>
                        <div><strong>Duration:</strong> {{ $loan->duration }} months</div>
                        <div><strong>Purpose:</strong> {{ $loan->purpose ?? '-' }}</div>

                    </div>

                    <div class="mt-3 flex flex-col sm:flex-row sm:space-x-2 space-y-2 sm:space-y-0">
                        <a href="{{ route('user.loans.show', $loan) }}"
                            class="flex-1 inline-flex items-center justify-center px-4 py-2 text-sm font-medium rounded-lg text-white bg-blue-600 hover:bg-blue-700 transition">
                            View
                        </a>

                    </div>

                </x-table.mobile-card>
            @empty
                <div class="px-6 py-6 text-center text-gray-400 dark:text-gray-500">
                    No loans found
                </div>
            @endforelse
        </x-slot>

        {{-- FOOTER / PAGINATION --}}
        {{-- Footer (pagination + perPage dropdown) --}}
        <x-slot name="footer">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between w-full gap-4 mt-6"
                x-data="{
                    open: false,
                    options: [5, 10, 15, 20, 25, 50],
                    selected: @js($perPage)
                }" x-init="$watch('selected', value => $wire.set('perPage', value));
                Livewire.on('perPageUpdated', event => selected = event.value);"
                x-transition:enter="transition-opacity ease-out duration-500" x-transition:enter-start="opacity-0"
                x-transition:enter-end="opacity-100" x-transition:leave="transition-opacity ease-in duration-300"
                x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0">

                <!-- Rows Per Page Selector -->
                <div class="relative text-sm w-fit">
                    <label class="block text-gray-700 dark:text-gray-300 mb-1 text-xs font-medium">
                        Rows per page
                    </label>

                    <button @click="open = !open"
                        class="w-36 bg-white dark:bg-gray-800 text-gray-800 dark:text-white border border-gray-300 dark:border-gray-600 rounded-md px-4 py-2 flex justify-between items-center shadow-sm hover:border-gray-400 dark:hover:border-gray-500 focus:outline-none focus:ring-2 focus:ring-sky-500"
                        type="button" aria-haspopup="listbox" aria-expanded="open">
                        <span x-text="selected"></span>
                        <svg class="w-4 h-4 ml-2 transform transition-transform duration-200"
                            :class="{ 'rotate-180': open }" fill="none" viewbox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M19 9l-7 7-7-7" />
                        </svg>
                    </button>

                    <!-- Dropdown -->
                    <div x-show="open" @click.outside="open = false"
                        x-transition:enter="transition ease-out duration-100"
                        x-transition:enter-start="opacity-0 scale-95" x-transition:enter-end="opacity-100 scale-100"
                        x-transition:leave="transition ease-in duration-75"
                        x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-95"
                        class="absolute bottom-0 mt-2 w-full z-50 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md shadow-lg">
                        <ul class="py-1 max-h-60 overflow-auto text-sm text-gray-800 dark:text-white">
                            <template x-for="option in options" :key="option">
                                <li>
                                    <a href="#" @click.prevent="selected = option; open = false"
                                        class="block px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700"
                                        :class="{ 'bg-gray-100 dark:bg-gray-700 font-semibold': selected === option }"
                                        x-text="option"></a>
                                </li>
                            </template>
                        </ul>
                    </div>
                </div>

                <!-- Pagination Links -->
                <div class="w-full md:w-auto text-center">{{ $this->loans->links() }}
                </div>

            </div>

        </x-slot>

    </x-table.wrapper>

</div>
