<?php

use App\Models\User;
use App\Services\OtpService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Livewire\Component;
use App\Services\LoginService;


new class extends Component {

    /**
     * OTP digits (6 inputs)
     */
    public array $otp = ['', '', '', '', '', ''];

    /**
     * Remaining OTP lifetime (seconds)
     */
    public int $expiresIn = 600;

    /**
     * Resend cooldown timer (seconds)
     */
    public int $resendCooldown = 30;

    /**
     * UI state flags
     */
    public bool $canResend = false;
    public bool $trustDevice = false; // default
    /**
     * Component boot
     */
    public function mount()
    {
        // OTP context must exist
        if (!session()->has('auth_email')) {
            return redirect()->route('login')->with([
                'message' => 'Login to receive OTP.',
                'type' => 'error',
            ]);
        }

        $now = now();

        /**
         * OTP expiration countdown
         */
        $otpExpiresAt = session('otp_expires_at');
        $this->expiresIn = $otpExpiresAt instanceof \Carbon\Carbon
            ? max(0, $now->diffInSeconds($otpExpiresAt, false))
            : 0;

        /**
         * Resend cooldown countdown
         */
        $cooldownStart = session('resend_cooldown_start');
        $this->resendCooldown = $cooldownStart instanceof \Carbon\Carbon
            ? max(0, 30 - $cooldownStart->diffInSeconds($now))
            : 0;

        $this->canResend = $this->resendCooldown <= 0;

        // OTP expired → allow resend immediately
        if ($this->expiresIn <= 0) {
            $this->expiresIn = 0;
            $this->canResend = true;
        }
    }

    /**
     * Resend OTP securely
     */
    public function resendOtp()
    {
        $email = session('auth_email');

        if (!$email) {
            return redirect()->route('login')->with([
                'message' => 'Session expired. Please restart verification.',
                'type' => 'error',
            ]);
        }

        try {
            app(OtpService::class)->generateAndSendOtp($email);

            session([
                'otp_expires_at' => now()->addMinutes(10),
                'resend_cooldown_start' => now(),
            ]);

            $this->expiresIn = 600;
            $this->resendCooldown = 30;
            $this->canResend = false;

            $this->dispatch('otpResent');
            $this->dispatch('showToast', message: 'OTP successfully resent.', type: 'success');
        } catch (\Throwable $e) {
            $this->addError('otp', 'Failed to resend OTP. Please try again.');
        }
    }

    /**
     * Cancel OTP verification
     */
    public function cancelVerification()
    {
        session()->forget([
            'auth_email',
            'otp_expires_at',
            'resend_cooldown_start',
            'otp_action',
            'remember_login',
        ]);

        return redirect()->route('login')->with([
            'message' => 'Verification cancelled.',
            'type' => 'success',
        ]);
    }

    /**
     * Generate secure device token
     */
    protected function generateDeviceToken(): string
    {
        return Str::uuid()->toString();
    }

    /**
     * Verify OTP and authenticate user
     */
    public function verifyOtp()
    {
        /*
        |--------------------------------------------------------------------------
        | 1. Ensure OTP session context exists
        |--------------------------------------------------------------------------
        */
        $email = session('auth_email');
        $otpAction = session('otp_action'); // login / register

        if (!$email || !$otpAction) {
            return redirect()->route('login')->with([
                'message' => 'Session expired. Please start over.',
                'type' => 'error',
            ]);
        }

        /*
        |--------------------------------------------------------------------------
        | 2. Validate OTP input
        |--------------------------------------------------------------------------
        */
        try {
            $this->validate([
                'otp' => ['required', 'array', 'size:6'],
                'otp.*' => ['required', 'digits:1'],
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('focusFirstInvalidOtp');
            throw $e;
        }

        $enteredOtp = implode('', $this->otp);

        /*
        |--------------------------------------------------------------------------
        | 3. Verify OTP via service
        |--------------------------------------------------------------------------
        */
        try {
            app(OtpService::class)->verifyOtp($email, $enteredOtp);
        } catch (\Throwable $e) {
            $this->addError('otp', $e->getMessage());
            return;
        }

        /*
        |--------------------------------------------------------------------------
        | 4. Authenticate + trusted device handling (transaction safe)
        |--------------------------------------------------------------------------
        */
        // inside verifyOtp()
        $user = DB::transaction(function () use ($email, $otpAction) {
            $user = User::where('email', $email)->firstOrFail();

            if ($otpAction === 'login') {
                // Delegate login tracking + trusted device to service
                app(LoginService::class)->recordLogin($user, $this->trustDevice);
            }

            return $user;
        });

        /*
        |--------------------------------------------------------------------------
        | 5. Clean OTP session state & Login User
        |--------------------------------------------------------------------------
        */
        Auth::login($user, session('remember_login', false));

        session()->forget([
            'auth_email',
            'otp_expires_at',
            'resend_cooldown_start',
            'remember_login',
            'otp_action',
        ]);

        /*
        |--------------------------------------------------------------------------
        | 6. Final redirect
        |--------------------------------------------------------------------------
        */
        // Decide message based on where user is going
        if ($user->isProfileComplete()) {
            session()->flash('message', 'Welcome back! You are now logged in.');
            session()->flash('type', 'success');
            $route = 'user.dashboard';
        } else {
            session()->flash('message', 'Almost there! Please complete your profile to get started.');
            session()->flash('type', 'info'); // can be different type if you like
            $route = 'onboarding';
        }

        // Redirect safely with Livewire
        return $this->redirectRoute($route, navigate: true);

    }
};
