<?php

use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use App\Traits\WithTableHelpers;
use App\Models\Loan;

new #[Layout('layouts::base', ['title' => 'Loans Listings'])] class extends Component {
    use WithPagination, WithTableHelpers;
    public $perPage = 10;
    public int $page = 1;
    public string $currencySymbol = '$';

    protected function searchableFields(): array
    {
        return ['id', 'user.name', 'user.email', 'amount', 'total_interest', 'status', 'created_at', 'first_due_date'];
    }

    public function delete($id)
    {
        Loan::findOrFail($id)->delete();
        $this->dispatch('showToast', message: 'Loan successfully deleted', type: 'success');
    }
    public function updatedPerPage()
    {
        $this->resetPage();
        $this->dispatch('perPageUpdated', $this->perPage);
    }

    #[Computed]
    public function loans()
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        $query = Loan::query()->with(['user']);

        // Apply visibility filter only if an admin is logged in
        if ($admin) {
            $query->visibleToAdmin($admin);
        } else {
            // No admin logged in, deny access
            $query->whereRaw('1 = 0');
        }

        $this->applySearch($query, $this->searchableFields());

        return $query->orderBy($this->sortField, $this->sortDirection)->paginate($this->perPage)->withQueryString();
    }
};
?>


<div class="space-y-5 py-5">
    <x-page-header title="Loans Management"
        subtitle="Overview of all user loans, their statuses, interest, repayment details, and transaction history."
        back-url="{{ route('admin.dashboard') }}">
    </x-page-header>

    <x-table.wrapper :items="$this->loans">

        {{-- SEARCH --}}
        <x-slot name="search">
            <div class="flex items-center justify-between gap-4">
                <div class="w-full md:w-1/3">
                    <div class="relative">
                        <x-flux::icon.magnifying-glass
                            class="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400 dark:text-gray-500" />
                        <input type="text" wire:model.live="search" placeholder="Search by name or email"
                            class="w-full pl-10 pr-4 py-2 text-sm rounded-md bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-sky-600 focus:border-sky-600 transition" />
                    </div>
                </div>
            </div>
        </x-slot>

        {{-- HEADERS --}}
        <x-slot name="head">
            <x-table.head label="Loan ID" field="id" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="User" />
            <x-table.head label="Amount" field="amount" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Interest" field="total_interest" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Created" field="created_at" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="First Due" field="first_due_date" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Status" field="status" sortable :sortField="$sortField" :sortDirection="$sortDirection" />
            <x-table.head label="Action" />
        </x-slot>

        {{-- BODY (DESKTOP) --}}
        <x-slot name="body">
            @forelse ($this->loans as $loan)
                @php
                    $user = $loan->user;
                    $account = $loan->account;
                    $profile = $account?->profile;
                    $currencySymbol = $profile ? \App\Support\Currency::symbol($profile->currency) : '$';

                    $statusClasses = match ($loan->status) {
                        'approved' => 'bg-green-500 text-white',
                        'pending' => 'bg-yellow-400 text-black',
                        'rejected' => 'bg-red-500 text-white',
                        'overdue' => 'bg-red-600 text-white',
                        'completed' => 'bg-blue-500 text-white',
                        'defaulted' => 'bg-gray-500 text-white',
                        default => 'bg-gray-300 text-black',
                    };
                @endphp

                <x-table.row wire:key="row-{{ $loan->id }}"
                    class="hover:shadow-lg transition-shadow duration-200 rounded-lg mb-2">

                    {{-- Loan ID --}}
                    <x-table.cell>{{ $loan->id ?? '_' }}</x-table.cell>

                    {{-- User --}}
                    <x-table.cell>
                        <a href="{{ route('admin.users.show', $user) }}" class="text-sm font-medium hover:underline">
                            {{ $user->name }} ({{ $user->email }})
                        </a>
                    </x-table.cell>

                    {{-- Amount --}}
                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->amount, 2) }}</x-table.cell>

                    {{-- Interest --}}
                    <x-table.cell>{{ $currencySymbol }}{{ number_format($loan->total_interest, 2) }}</x-table.cell>

                    {{-- Created date --}}
                    <x-table.cell>{{ $loan->created_at?->format('d M Y') ?? '-' }}</x-table.cell>

                    {{-- First Due --}}
                    <x-table.cell>{{ $loan->first_due_date?->format('d M Y') ?? '-' }}</x-table.cell>

                    {{-- Status --}}
                    <x-table.cell>
                        <span class="px-2 py-1 rounded-full text-xs {{ $statusClasses }}">
                            {{ ucfirst($loan->status) }}
                        </span>
                    </x-table.cell>

                    {{-- Actions --}}
                    <x-table.cell>
                        <a href="{{ route('admin.loans.show', $loan) }}"
                            class="inline-flex items-center px-3 py-1.5 text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 transition">
                            View
                        </a>
                        <button x-data
                            @click.prevent="if(confirm('Are you sure you want to delete this loan?')) { $wire.delete({{ $loan->id }}) }"
                            class="inline-flex items-center px-3 py-1.5 text-sm font-medium rounded-md text-white bg-red-600 hover:bg-red-700 transition">
                            Delete
                        </button>
                    </x-table.cell>

                </x-table.row>
            @empty
                <tr>
                    <td colspan="8" class="px-6 py-10 text-center text-gray-400 dark:text-gray-500">
                        No loans found
                    </td>
                </tr>
            @endforelse
        </x-slot>

        {{-- MOBILE --}}
        <x-slot name="mobile">
            @forelse ($this->loans as $loan)
                @php
                    $user = $loan->user;
                    $account = $loan->account;
                    $profile = $account?->profile;
                    $currencySymbol = $profile ? \App\Support\Currency::symbol($profile->currency) : '$';

                    $statusClasses = match ($loan->status) {
                        'approved' => 'bg-green-100 text-green-800',
                        'pending' => 'bg-yellow-100 text-yellow-800',
                        'rejected' => 'bg-red-100 text-red-800',
                        'overdue' => 'bg-red-200 text-red-900',
                        'completed' => 'bg-blue-100 text-blue-800',
                        'defaulted' => 'bg-gray-200 text-gray-800',
                        default => 'bg-gray-100 text-gray-700',
                    };
                @endphp

                <x-table.mobile-card class="rounded-xl shadow-md p-4 mb-4 hover:shadow-xl transition duration-200">

                    <div class="flex flex-col text-sm text-gray-700 dark:text-gray-300 space-y-1">
                        <div><strong>Loan ID:</strong> {{ $loan->id ?? '_' }}</div>
                        <div><strong>User:</strong> {{ $user->name }} ({{ $user->email }})</div>
                        <div><strong>Amount:</strong> {{ $currencySymbol }}{{ number_format($loan->amount, 2) }}</div>
                        <div><strong>Interest:</strong>
                            {{ $currencySymbol }}{{ number_format($loan->total_interest, 2) }}</div>
                        <div><strong>Created:</strong> {{ $loan->created_at?->format('d M Y') ?? '-' }}</div>
                        <div><strong>First Due:</strong> {{ $loan->first_due_date?->format('d M Y') ?? '-' }}</div>
                        <div><strong>Status:</strong>
                            <span
                                class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $statusClasses }}">
                                {{ ucfirst($loan->status) }}
                            </span>
                        </div>
                    </div>

                    <div class="mt-3 flex flex-col sm:flex-row sm:space-x-2 space-y-2 sm:space-y-0">
                        <a href="{{ route('admin.loans.show', $loan) }}"
                            class="flex-1 inline-flex items-center justify-center px-4 py-2 text-sm font-medium rounded-lg text-white bg-blue-600 hover:bg-blue-700 transition">
                            View
                        </a>
                        <button x-data
                            @click.prevent="if(confirm('Are you sure you want to delete this loan?')) { $wire.delete({{ $loan->id }}) }"
                            class="flex-1 inline-flex items-center justify-center px-4 py-2 text-sm font-medium rounded-lg text-white bg-red-600 hover:bg-red-700 transition">
                            Delete
                        </button>
                    </div>

                </x-table.mobile-card>
            @empty
                <div class="px-6 py-6 text-center text-gray-400 dark:text-gray-500">
                    No loans found
                </div>
            @endforelse
        </x-slot>

        {{-- FOOTER / PAGINATION --}}
        <x-slot name="footer">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between w-full gap-4 mt-6">
                <div class="w-full md:w-auto text-center">{{ $this->loans->links() }}</div>
            </div>
        </x-slot>

    </x-table.wrapper>

</div>
