<?php

use Livewire\Component;
use App\Models\Contact;
use Livewire\WithFileUploads;
use App\Notifications\ContactCreated;
use App\Services\AdminNotifier;

new class extends Component {
       use WithFileUploads;

    public $attachments = [];
    public $subject = '';
    public $email = '';
    public $phone = '';
    public $message = '';
    public $fullname = '';

    // Validation rules
    protected function rules()
    {
        return [
            'subject' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'fullname' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20|regex:/^[0-9+\-\(\)\s]+$/',
            'attachments' => 'nullable|array|max:5',
            'attachments.*' => 'file|max:5120|mimes:jpg,jpeg,png,webp,pdf,doc,docx,xls,xlsx',
            'message' => 'required|string|min:10|max:5000',
        ];
    }

    // Custom messages
    protected $messages = [
        'subject.required' => 'Please enter a subject for your message.',
        'subject.string' => 'Subject must be valid text.',
        'subject.max' => 'Subject cannot exceed 255 characters.',

        'email.required' => 'Please enter your email address.',
        'email.email' => 'Please enter a valid email address.',
        'email.max' => 'Email cannot exceed 255 characters.',

        'fullname.required' => 'Please enter your full name.',
        'fullname.string' => 'Full name must be valid text.',
        'fullname.max' => 'Full name cannot exceed 255 characters.',

        'phone.string' => 'Phone number must be valid text.',
        'phone.max' => 'Phone number cannot exceed 20 characters.',
        'phone.regex' => 'Phone number contains invalid characters.',

        'attachments.array' => 'Attachments must be an array of files.',
        'attachments.max' => 'You can upload a maximum of 5 attachments.',
        'attachments.*.file' => 'Each attachment must be a valid file.',
        'attachments.*.max' => 'Each attachment must not exceed 5MB.',
        'attachments.*.mimes' => 'Allowed file types: jpg, jpeg, png, webp, pdf, doc, docx, xls, xlsx.',

        'message.required' => 'Please enter your message.',
        'message.string' => 'Message must be valid text.',
        'message.min' => 'Message must be at least 10 characters.',
        'message.max' => 'Message cannot exceed 5000 characters.',
    ];

    // Real-time validation
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    // Limit attachments
    public function updatedAttachments()
    {
        if (count($this->attachments) > 5) {
            $this->attachments = array_slice($this->attachments, 0, 5);
            $this->addError('attachments', 'You can upload a maximum of 5 attachments.');
        }
    }

    // Remove attachment
    public function removeAttachment($index)
    {
        unset($this->attachments[$index]);
        $this->attachments = array_values($this->attachments);
    }

    // Submit form
   public function submit()
{
    $this->validate();

    // Store attachments
    $attachmentPaths = [];
    if ($this->attachments) {
        foreach ($this->attachments as $file) {
            $attachmentPaths[] = $file->store('contacts', 'public');
        }
    }

    // Save contact
    $contact = Contact::create([
        'fullname' => $this->fullname,
        'email' => $this->email,
        'phone' => $this->phone,
        'subject' => $this->subject,
        'message' => $this->message,
        'attachments' => $attachmentPaths,
    ]);

    // ✅ Notify guest
    Notification::route('mail', $this->email)
        ->notify(new ContactCreated($contact));

    // Notify admins
    AdminNotifier::notify('New Complaint', [
        'name' => $this->fullname,
        'email' => $this->email,
        'phone' => $this->phone,
        'subject' => $this->subject,
        'message' => $this->message,
        'attachments' => $attachmentPaths,
    ]);

    // Reset form
    $this->reset(['subject', 'attachments', 'message', 'fullname', 'email', 'phone']);

    // Dispatch frontend events
    $this->dispatch('notification-sent');
    $this->dispatch('showToast', message: 'Your message has been submitted successfully.', type: 'success');
}


};
?>

<div>
	<div class="py-16 px-4 bg-gray-100 dark:bg-gray-900">
		<div
			class="max-w-7xl mx-auto grid md:grid-cols-2 gap-8 items-stretch">
			<!-- Contact Form -->
			<div class="bg-white dark:bg-gray-800 shadow-xl rounded-xl p-8">
				<form wire:submit.prevent='submit' enctype="multipart/form-data">
					<h2 class="text-2xl font-bold uppercase text-[#034460] dark:text-white tracking-wide mb-6">Get in
						                        Touch
					</h2>

					<div class="grid gap-5">
						<div>
							<x-input wire:model.live.blur='subject' label="Subject" placeholder="Enter subject"/>
						</div>
						<div>
							<x-input wire:model.live.blur='fullname' label="Full name" placeholder="Enter full name"/>
						</div>
						<div>
							<x-input wire:model.live.blur='email' type="email" label="Email address" placeholder="Enter enter address"/>
						</div>
						<div>
							<x-input wire:model.live.blur='phone' type="number" label="Phone Number" placeholder="Enter reachable phone number"/>
						</div>
						<div>
							<x-textarea wire:model.live.blur='message' rows="5" label="Message" placeholder="Additional information or instructions..."/>
						</div>

						{{-- Attachment --}}
						<div>
							@include('partials.support-upload')
						</div>
					</div>


					<div class="mt-6">
						<x-button type="submit">
							Send Message
						</x-button>
					</div>
				</form>
			</div>

			<!-- Contact Info -->
			<div
				class="relative rounded-xl overflow-hidden p-8">
				<!-- Background Image -->
				<img
				src="{{ asset('images/home/feature1.jpg') }}" alt="Background" class="absolute inset-0 w-full h-full object-cover z-0"/>

				<!-- Gradient Overlay -->
				<div class="absolute inset-0 bg-linear-to-t from-sky-800 to-sky-900 opacity-95 z-10"></div>

				<!-- SVG Curve -->
				<div class="relative z-20 hidden md:flex">
					<svg class="w-full h-28 lg:h-36 xl:h-40" viewbox="0 0 1440 200" fill="none" xmlns="http://www.w3.org/2000/svg" preserveaspectratio="none">
						<path fill="#ffffff" d="M0,100 C300,220 1000,0 1440,160 L1440,0 L0,0 Z"/>
					</svg>
				</div>

				<!-- Info Box -->
				<div class="relative z-30 w-full md:-mt-20 lg:-mt-28 xl:-mt-32 p-6 md:p-8 space-y-8 md:space-y-10
						           md:w-[95%] mx-auto flex flex-col justify-center
						           bg-white dark:bg-gray-800 shadow-2xl rounded-b-xl
						           text-[#022f42] dark:text-white max-w-screen-sm sm:max-w-3xl">

					<div>
						<h2 class="text-xl sm:text-2xl font-bold uppercase tracking-wide mb-2">Contact Details</h2>
						<p class="text-sm sm:text-base font-medium tracking-wide">Need consultation? Reach out to us
							                            anytime.</p>
					</div>
					<div class="space-y-6">
						<div class="flex items-center gap-4">
							<div class="w-12 h-12 flex items-center justify-center rounded-full bg-sky-800 shrink-0">
								<x-flux::icon.phone class="w-5 h-5 text-white"/>
							</div>
							<div>
								<h4 class="text-base md:text-lg font-semibold">Phone</h4>
								<a href="tel:{{ config('app.phone') }}" class="block text-sm md:text-base font-medium dark:text-gray-400">{{
    config('app.phone') }}</a>
							</div>
						</div>

						<div class="flex items-center gap-4">
							<div class="w-12 h-12 flex items-center justify-center rounded-full bg-sky-800 shrink-0">
								<x-flux::icon.envelope class="w-5 h-5 text-white"/>
							</div>
							<div>
								<h4 class="text-base md:text-lg font-semibold">Email</h4>
								<a
									href="mailto:{{ config('app.email') }}" class="block text-sm md:text-base font-medium break-all md:break-normal dark:text-gray-400">{{ config('app.email') }}
								</a>
							</div>
						</div>

						<div class="flex items-center gap-4">
							<div class="w-12 h-12 flex items-center justify-center rounded-full bg-sky-800 shrink-0">
								<x-flux::icon.map-pin class="w-5 h-5 text-white"/>
							</div>
							<div>
								<h4 class="text-base md:text-lg font-semibold">Location</h4>
								<p class="text-sm md:text-base font-medium dark:text-gray-400">{{
    config('app.app_address') }}</p>
							</div>
						</div>

					</div>
					<!-- Login -->
					<div class="flex flex-col items-center p-6 bg-white dark:bg-gray-700 rounded-lg shadow hover:shadow-lg transition">
						<x-flux::icon.user-circle class="w-10 h-10 text-sky-600 mb-3"/>
						<h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">Login to Your Account</h3>
						<p class="text-gray-600 dark:text-gray-300 text-sm mb-4 text-center">
							Access your dashboard and manage support requests quickly and securely.
						</p>
						<a href="{{ route('login') }}" class="inline-flex items-center gap-2 px-4 py-2 bg-sky-800 hover:bg-sky-900 text-white rounded-md font-semibold transition">
							Login
							<x-flux::icon.arrow-right class="w-4 h-4"/>
						</a>
					</div>
				</div>


			</div>
		</div>
	</div>

	<!-- Map Section -->
	<section class="w-full min-h-125 fade-in-up bg-gray-100 dark:bg-gray-900 px-6">
		<div class="max-w-7xl mx-auto">
			<iframe src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d2482.9647948929364!2d-0.08488802279203012!3d51.51386187181486!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x48760352a1c1f1cf%3A0x6472b865c5004363!2sThe%20Leadenhall%20Building!5e0!3m2!1sen!2sgh!4v1748910305053!5m2!1sen!2sgh" width="100%" height="450" style="border:0;" allowfullscreen="" loading="lazy" referrerpolicy="no-referrer-when-downgrade"></iframe>
		</div>
	</section>
</div>
