<?php
namespace App\Services;


use App\Models\Account;
use App\Models\AccountCryptoBalance;
use App\Models\CryptoFiatSwap;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;

class CryptoFiatSwapService
{
    /**
     * Swap crypto to fiat
     *
     * @param Account $account - User's fiat account
     * @param AccountCryptoBalance $crypto - User's crypto balance
     * @param string $cryptoAmount - Amount of crypto to swap
     * @param string $rate - Conversion rate: 1 crypto = $rate fiat
     * @param string $idempotencyKey - Unique request key
     */
    public function swap(
        Account $account,
        AccountCryptoBalance $crypto,
        string $cryptoAmount,
        string $rate,
        string $idempotencyKey
    ) {
        if (CryptoFiatSwap::where('idempotency_key', $idempotencyKey)->exists()) {
            return;
        }

        DB::transaction(function () use ($account, $crypto, $cryptoAmount, $rate, $idempotencyKey) {
            $fiatAmount = bcmul($cryptoAmount, $rate, 2);

            $crypto->lock($cryptoAmount);

            $swap = CryptoFiatSwap::create([
                'idempotency_key' => $idempotencyKey,
                'account_id' => $account->id,
                'crypto_balance_id' => $crypto->id,
                'crypto_code' => $crypto->code,
                'network' => $crypto->network,
                'crypto_amount' => $cryptoAmount,
                'rate' => $rate,
                'fiat_amount' => $fiatAmount,
                'status' => 'pending',
            ]);

            $account->balance = (string) bcadd((string) $account->balance, $fiatAmount, 2);
            $account->save();

            $crypto->finalize($cryptoAmount);

            Transaction::create([
                'account_id' => $account->id,
                'crypto_balance_id' => $crypto->id,
                'type' => 'swap_credit',
                'amount' => $fiatAmount,
                'balance_after' => $account->available_balance,
                'status' => Transaction::STATUS_COMPLETED,
            ]);

            Transaction::create([
                'account_id' => null,                       // ✅ NOT a fiat account
                'crypto_balance_id' => $crypto->id,         // ✅ crypto ledger
                'type' => 'swap_debit',
                'amount' => bcmul($cryptoAmount, '-1', 8),  // ✅ crypto unit
                'balance_after' => $crypto->available_balance,
                'status' => Transaction::STATUS_COMPLETED,
            ]);

            $swap->update(['status' => 'completed']);
        });
    }

    /**
     * Swap fiat to crypto
     */
    public function swapToCrypto(
        Account $account, // Fiat account from which to debit
        AccountCryptoBalance $crypto, // Crypto wallet to credit
        string $fiatAmount,
        string $rate,
        string $idempotencyKey
    ) {
        if (CryptoFiatSwap::where('idempotency_key', $idempotencyKey)->exists()) {
            return;
        }

        DB::transaction(function () use ($account, $crypto, $fiatAmount, $rate, $idempotencyKey) {
            $cryptoAmount = bcdiv($fiatAmount, $rate, 8); // Calculate crypto from fiat

            if (bccomp($account->available_balance, $fiatAmount, 2) < 0) {
                throw new \Exception('Insufficient fiat balance.');
            }

            // Lock fiat from account
            $account->balance = (string) bcsub((string) $account->balance, $fiatAmount, 2);
            $account->save();

            $crypto->credit($cryptoAmount); // Method to increase crypto balance

            $swap = CryptoFiatSwap::create([
                'idempotency_key' => $idempotencyKey,
                'account_id' => $account->id,
                'crypto_balance_id' => $crypto->id,
                'crypto_code' => $crypto->code,
                'network' => $crypto->network,
                'crypto_amount' => $cryptoAmount,
                'rate' => $rate,
                'fiat_amount' => $fiatAmount,
                'status' => 'pending',
            ]);

            Transaction::create([
                'account_id' => $account->id,
                'type' => 'swap_debit',
                'amount' => bcmul($fiatAmount, '-1', 2),
                'balance_after' => $account->available_balance,
                'status' => Transaction::STATUS_COMPLETED,
            ]);

            Transaction::create([
                'account_id' => null,                      // ✅ crypto ledger only
                'crypto_balance_id' => $crypto->id,
                'type' => 'swap_credit',
                'amount' => $cryptoAmount,                 // crypto unit
                'balance_after' => $crypto->available_balance,
                'status' => Transaction::STATUS_COMPLETED,
            ]);


            $swap->update(['status' => 'completed']);
        });
    }
}
