<?php

namespace App\Notifications;

use App\Models\TaxRefund;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;

class RefundStatusNotification extends Notification
{

    protected TaxRefund $refund;
    protected string $status;
    protected ?string $note;

    /**
     * Create a new notification instance.
     */
    public function __construct(TaxRefund $refund, string $status, ?string $note = null)
    {
        $this->refund = $refund;
        $this->status = $status;
        $this->note = $note;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable)
    {
        return ['mail', 'database']; // Send via email and store in database
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable)
    {
        return (new MailMessage)
            ->view('emails.refund-status', [
                'refund' => $this->refund,
                'status' => $this->status,
                'note' => $this->note,
            ]);
    }

    /**
     * Store notification in database for in-app display.
     */
    public function toArray($notifiable)
    {
        $statusTitle = ucfirst(str_replace('_', ' ', $this->status));

        return [
            'refund_id' => $this->refund->id,
            'status' => $this->status,
            'amount' => $this->refund->formattedAmount(), // uses user currency
            'submitted_at' => $this->refund->created_at,

            // Fields for in-app display
            'title' => "Tax Refund {$statusTitle}",
            'message' => "Your tax refund request of {$this->refund->formattedAmount()} submitted on " .
                "{$this->refund->created_at->format('M d, Y')} is now {$statusTitle}.",
            'link' => route('user.tax.refund.track'),
        ];
    }


}
