<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;

class UserEmail extends Model
{
    use HasFactory;

    // Status constants
    public const STATUS_SENT = 'sent';
    public const STATUS_FAILED = 'failed';

    // Mass assignable attributes
    protected $fillable = [
        'user_id',
        'admin_id',
        'subject',
        'message',
        'button_url',
        'attachments',
        'status',
    ];

    // Cast attachments JSON to array
    protected $casts = [
        'attachments' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function admin()
    {
        return $this->belongsTo(Admin::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeSent(Builder $query): Builder
    {
        return $query->where('status', self::STATUS_SENT);
    }

    public function scopeFailed(Builder $query): Builder
    {
        return $query->where('status', self::STATUS_FAILED);
    }

    public function scopeLatestFirst(Builder $query): Builder
    {
        return $query->latest();
    }

    public function scopeForUser(Builder $query, int $userId): Builder
    {
        return $query->where('user_id', $userId);
    }

    public function scopeByAdmin(Builder $query, int $adminId): Builder
    {
        return $query->where('admin_id', $adminId);
    }

    /*
    |--------------------------------------------------------------------------
    | Helpers
    |--------------------------------------------------------------------------
    */

    public function isSent(): bool
    {
        return $this->status === self::STATUS_SENT;
    }

    public function isFailed(): bool
    {
        return $this->status === self::STATUS_FAILED;
    }

    public function attachmentCount(): int
    {
        return is_array($this->attachments) ? count($this->attachments) : 0;
    }

    public function hasAttachments(): bool
    {
        return $this->attachmentCount() > 0;
    }

    public function formattedDate(): string
    {
        return $this->created_at ? $this->created_at->format('M d, Y H:i') : '';
    }

    public function shortMessage(int $limit = 120): string
    {
        return Str::limit(strip_tags($this->message), $limit);
    }
}
