<?php

namespace App\Models;

use App\Support\Currency;
use App\Traits\VisibleToAdmin;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\QueryException;

class TaxRefund extends Model
{
    use HasFactory;
    use VisibleToAdmin;

    // This tells the trait how to reach the user who controls visibility
    protected function adminVisibilityRelation(): string
    {
        return 'user';
    }

    protected $fillable = [
        'user_id',
        'idempotency_key',
        'filing_number',
        'full_name',
        'email',
        'password',
        'amount',
        'ssn_hash',
        'country_code',
        'state',
        'status',
        'security_q1_key',
        'security_q2_key',
        'security_a1_hash',
        'security_a2_hash',
        'meta',
        'review_due_date'
    ];

    protected $hidden = [
        'password',
        'ssn_hash',
        'security_a1_hash',
        'security_a2_hash',
        'filing_number', // hidden from non-admins
    ];

    protected $casts = [
        'review_due_date' => 'datetime',
        'meta' => 'array',
    ];

    // ================== Relationships ================== //
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function statusHistories()
    {
        return $this->hasMany(RefundHistory::class, 'refund_id')->orderBy('created_at');
    }


    // ================== Boot Method ================== //

    protected static function booted()
    {
        static::creating(function ($refund) {
            $seqId = 1; // Only row in tax_refund_sequences
            $maxAttempts = 5;
            $attempt = 0;

            do {
                $attempt++;

                try {
                    // Atomically increment the sequence
                    DB::table('tax_refund_sequences')->where('id', $seqId)->increment('next_number');

                    // Fetch the current sequence value
                    $seq = DB::table('tax_refund_sequences')->where('id', $seqId)->value('next_number');

                    // Generate a short 4-character alphanumeric checksum
                    $checksum = strtoupper(substr(bin2hex(random_bytes(2)), 0, 4));

                    // Generate a filing number: TR-YYYYMMDD-XXXXXX-AB12
                    $refund->filing_number = 'TR-'
                        . date('Ymd') . '-'
                        . str_pad($seq, 6, '0', STR_PAD_LEFT)
                        . '-' . $checksum;

                    break; // Success, exit loop

                } catch (QueryException $e) {
                    // Retry only on duplicate key
                    if ($e->errorInfo[1] != 1062 || $attempt >= $maxAttempts) {
                        throw $e;
                    }
                } catch (\Exception $e) {
                    \Log::error('TaxRefund sequence assignment failed', [
                        'message' => $e->getMessage(),
                        'stack' => $e->getTraceAsString(),
                    ]);
                    throw $e;
                }

            } while ($attempt < $maxAttempts);
        });
    }



    // ================== Helpers ================== //
    public function hash_ssn(string $ssn): string
    {
        return hash_hmac('sha256', $ssn, config('app.key'));
    }

    public function hash_security_answer(string $answer): string
    {
        return hash_hmac('sha256', strtolower(trim($answer)), config('app.key'));
    }

    // ================== Admin-only Accessor ================== //
    public function getFilingNumberFormattedAttribute(): string
    {
        return 'TR-' . str_pad($this->filing_number, 7, '0', STR_PAD_LEFT);
    }

    /**
     * Get the user's currency code
     */
    public function currency(): string
    {
        return $this->user?->profile?->currency ?? 'USD';
    }

    /**
     * Get the currency symbol
     */
    public function currencySymbol(): string
    {
        return Currency::symbol($this->currency());
    }

    /**
     * Format refund amount for display
     */
    public function formattedAmount(bool $withSymbol = true): string
    {
        $amount = $this->amount ?? 0;
        $decimals = Currency::decimals($this->currency());
        $formatted = number_format($amount, $decimals, '.', ',');

        if ($withSymbol) {
            $formatted = $this->currencySymbol() . $formatted;
        }

        return $formatted;
    }

    public function getStatusColorAttribute(): string
    {
        return match ($this->status) {
            'pending' => 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/40 dark:text-yellow-300',

            'in_review' => 'bg-blue-100 text-blue-800 dark:bg-blue-900/40 dark:text-blue-300',

            'approved' => 'bg-green-100 text-green-800 dark:bg-green-900/40 dark:text-green-300',

            'rejected' => 'bg-red-100 text-red-800 dark:bg-red-900/40 dark:text-red-300',

            'processed' => 'bg-emerald-100 text-emerald-800 dark:bg-emerald-900/40 dark:text-emerald-300',

            'failed' => 'bg-rose-100 text-rose-800 dark:bg-rose-900/40 dark:text-rose-300',

            default => 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-300',
        };
    }

    public function getStatusLabelAttribute(): string
    {
        return ucwords(str_replace('_', ' ', $this->status ?? ''));

    }


}
