<?php

namespace App\Models;

use App\Traits\VisibleToAdmin;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Grant extends Model
{
    use HasFactory;
    use VisibleToAdmin;

    // This tells the trait how to reach the user who controls visibility
    protected function adminVisibilityRelation(): string
    {
        return 'user';
    }
    protected $fillable = [
        'type',
        'user_id',
        'amount',
        'purposes',
        'metadata',
        'status',
    ];

    protected $casts = [
        'purposes' => 'array',
        'metadata' => 'array',
        'amount' => 'decimal:2',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }
    public function grantLogs()
    {
        return $this->hasMany(GrantLog::class);
    }

    /**
     * Status labels for display
     */
    public const STATUS_LABELS = [
        'pending' => 'Submitted',
        'in_review' => 'In Review',
        'approved' => 'Approved',
        'rejected' => 'Rejected',
        'processed' => 'Processed',
        'on_hold' => 'On Hold',
        'failed' => 'Failed',
    ];

    /**
     * Status colors for badges
     */
    public const STATUS_COLORS = [
        'pending' => 'bg-yellow-100 dark:bg-yellow-600 text-yellow-800 dark:text-black text-xs',
        'in_review' => 'bg-yellow-200 dark:bg-yellow-500 text-yellow-900 dark:text-black text-xs',
        'approved' => 'bg-green-100 dark:bg-green-600 text-green-800 dark:text-white text-xs',
        'rejected' => 'bg-red-100 dark:bg-red-600 text-red-800 dark:text-white text-xs',
        'processed' => 'bg-indigo-100 dark:bg-indigo-600 text-indigo-800 dark:text-white text-xs',
        'on_hold' => 'bg-gray-200 dark:bg-gray-600 text-gray-800 dark:text-white text-xs',
        'failed' => 'bg-gray-300 dark:bg-gray-700 text-gray-900 dark:text-white text-xs',
    ];


    /**
     * Get display label for status
     */
    public function getStatusLabelAttribute(): string
    {
        return self::STATUS_LABELS[$this->status] ?? ucfirst(str_replace('_', ' ', $this->status));
    }

    /**
     * Get badge color class for status
     */
    public function getStatusColorAttribute(): string
    {
        return self::STATUS_COLORS[$this->status] ?? 'bg-gray-400';
    }
}
