<?php

namespace App\Livewire\Forms;

use Livewire\Attributes\Validate;
use Livewire\Form;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;


class OnboardingForm extends Form
{
    public string $name = '';
    public string $username = '';
    public string $date_of_birth = '';
    public string $gender = '';
    public string $marital_status = '';
    public string $occupation = '';
    public string $dial_code = '';
    public string $city = '';
    public string $zip_code = '';
    public string $address_1 = '';
    public string $address_2 = '';
    public string $phone = '';
    public ?string $referral_code = null;

    public string $currency = '';
    public string $bank_account_type = '';
    public string $password = '';
    public string $password_confirmation = '';

    public bool $agreement_accepted = false;

    /* ---------- Validation ---------- */

    public function rules(int $step, array $states = []): array
    {
        return match ($step) {
            1 => [
                'name' => 'required|string|max:255',
                'username' => [
                    'required',
                    'string',
                    'max:30',
                    'regex:/^[A-Za-z0-9_]+$/',
                    Rule::unique('profiles', 'username')->ignore(auth()->id(), 'user_id'),
                ],
                'date_of_birth' => 'required|date|before:today',
                'gender' => 'required|in:male,female,others',
                'marital_status' => 'required|in:single,married,divorced,widowed,others',
                'occupation' => 'required|string|max:255',
                'referral_code' => [
                    'nullable',
                    Rule::exists('users', 'referral_code'),
                    function ($attribute, $value, $fail) {
                            $user = auth()->user();
                            if ($user && $user->referral_code === $value) {
                                $fail('You cannot use your own referral code.');
                            }
                        }
                ],
            ],

            2 => [
                'address_1' => 'required|string|max:255',
                'address_2' => 'nullable|string|max:255',
                'city' => 'required|string|max:100',
                'zip_code' => 'required|string|max:20',
                'phone' => 'required|string|regex:/^\+?[0-9\s\-]{7,20}$/',
                'dial_code' => 'required|string|max:5',
                'state' => $states
                    ? ['required', Rule::in(array_column($states, 'name'))]
                    : 'required|string',
                'country' => 'required|string|max:3',
            ],

            3 => [
                'bank_account_type' => [
                    'required',
                    Rule::in(array_keys(config('bank.account_types')))
                ],
                'currency' => 'required|string|size:3',
                'password' => [
                    'required',
                    'confirmed',
                    Password::min(8)
                        ->letters()
                        ->mixedCase()
                        ->numbers()
                        ->symbols()
                        ->uncompromised(),
                ],
            ],

            4 => [
                'agreement_accepted' => 'accepted',
            ],

            default => []
        };
    }

    public function messages(int $step): array
    {
        return match ($step) {
            1 => [
                'name.required' => 'Please enter your full name.',
                'username.required' => 'A username is required.',
                'username.max' => 'Usernames cannot exceed 30 characters.',
                'username.regex' => 'Username may only contain letters, numbers, and underscores.',
                'username.unique' => 'This username is already taken.',
                'date_of_birth.required' => 'Please provide your date of birth.',
                'date_of_birth.before' => 'Date of birth must be in the past.',
                'gender.required' => 'Please select your gender.',
                'marital_status.required' => 'Please select your marital status.',
                'occupation.required' => 'Please tell us your occupation.',
            ],

            2 => [
                'phone.required' => 'Your phone number is required.',
                'phone.regex' => 'Please enter a valid phone number. Include country code if needed.',
                'dial_code.required' => 'Please select your country code.',
                'dial_code.max' => 'Invalid country code.',
                'country.required' => 'Please select your country.',
                'state.required' => 'Please select a state/province.',
                'address_1.required' => 'Street address line 1 is required.',
                'city.required' => 'City is required.',
                'zip_code.required' => 'Postal / ZIP code is required.',
            ],

            3 => [
                'bank_account_type.required' => 'Please select a bank account type.',
                'bank_account_type.in' => 'Invalid bank account type selected.',
                'currency.required' => 'Please select a currency.',
                'currency.string' => 'Invalid currency format.',
                'currency.size' => 'Currency code must be 3 letters (e.g. USD, EUR).',
                'password.required' => 'Please create a password.',
                'password.confirmed' => 'Passwords do not match.',
                'password.min' => 'Password must be at least 8 characters.',
                'password.letters' => 'Include at least one letter.',
                'password.mixed' => 'Use both uppercase and lowercase letters.',
                'password.numbers' => 'Include at least one number.',
                'password.symbols' => 'Include at least one symbol.',
                'password.*.uncompromised' => 'This password has appeared in a data leak. Please choose another.',
            ],

            4 => [
                'agreement_accepted.accepted' => 'You must agree to the Terms and Privacy Policy to continue.',
            ],

            default => []
        };
    }
    public function fillFromArray(array $data): void
    {
        foreach ($data as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    public function toArray(): array
    {
        return collect(get_object_vars($this))
            ->except(['password_confirmation'])
            ->toArray();
    }
}

